<?php


namespace Cms\LinkResource\Provider\CMS;

use Cms\Client\Article\ArticleClient;
use Cms\Client\Article\CategoryClient;
use Cms\Client\Service\ArticleService;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\ProviderClientAdapterInterface;
use Cms\LinkResource\Provider\ProviderRestfulClientAdapter;
use Cms\LinkResource\Provider\SearchAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareTrait;
use Cms\Model\Article\ArticleProfil;
use Cms\Search\Article\ArticleSearchClient;

/**
 * Class CmsArticleProvider
 * @package Cms\LinkResource\Provider\CMS
 */
class CmsArticleProvider extends AbstractCmsArticleProvider implements
    SearchAdapterAwareInterface
{
    use SearchAdapterAwareTrait;

    /** @var \Cms\Client\Service\ArticleService */
    private $articleService;

    /** @var \Cms\Search\Article\ArticleSearchClient|null */
    private $searchClient;

    /**
     * CmsArticleProvider constructor.
     * @param ArticleService $articleService
     * @param ArticleSearchClient|null $searchClient
     */
    public function __construct(
        ArticleService $articleService,
        ArticleSearchClient $searchClient = null
    ) {
        $this->articleService = $articleService;
        $this->searchClient = $searchClient;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @return CategoryClient|ArticleClient|null
     */
    protected function getClientByTypeCode($typeCode)
    {
        switch ((string)$typeCode) {
            case LinkResourceType::GALLERY:
            case LinkResourceType::ARTICLE:
                return $this->articleService->getArticleClient();
                break;
            case LinkResourceType::ARTICLE_CATEGORY:
                return $this->articleService->getCategoryClient();
                break;
        }
        return null;
    }

    /**
     * @param mixed $typeCode
     * @return \Cms\LinkResource\Provider\ProviderClientAdapterInterface
     * @throws \RuntimeException
     */
    protected function getClientAdapter($typeCode) : ProviderClientAdapterInterface
    {
        $client = $this->getClientByTypeCode($typeCode);
        if ($client !== null) {
            return new ProviderRestfulClientAdapter($client);
        }
        throw new \RuntimeException('no client adapter');
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param mixed $query
     * @return array
     */
    public function execSearch($typeCode, $query)
    {
        // recherche par id
        if (is_numeric($query)) {
            $resource = $this->getData($typeCode, $query);
            if ($resource) {
                return [
                    $resource->id,
                ];
            }
        }
        switch ((string)$typeCode) {
            case LinkResourceType::ARTICLE:
            case LinkResourceType::GALLERY:
                if (null === $this->searchClient) {
                    return [];
                }
                if ((string)$typeCode === LinkResourceType::GALLERY) {
                    $query .= ' AND profil:' . ArticleProfil::CMS_GALERIE;
                }
                /*if ((string)$typeCode === LinkResourceType::ARTICLE) {
                    $query .= " AND NOT profil:" . ArticleProfil::CMS_GALERIE;
                }*/
                if (null !== $this->scopeId) {
                    $query .= ' AND scope_id:' . $this->scopeId;
                }
                $articles = $this->searchClient->search($query);
                // On récupère les id de chaque article pour les renvoyer
                $articles = iterator_to_array($articles);
                foreach ($articles as &$article) {
                    $article = $article->getId();
                }
                return $articles;
                break;
            case LinkResourceType::ARTICLE_CATEGORY:
                $results = $this->getClientByTypeCode($typeCode)
                    ->getIndexBySearch($query, $this->scopeId);
                if (empty($results)) {
                    return [];
                }
                // On récupère les id de chaque modele pour les renvoyer
                $results = iterator_to_array($results);
                foreach ($results as &$model) {
                    $model = $model->id;
                }
                return $results;
                break;
        }
        return [];
    }
}
