<?php


namespace Move\Config;

use League\Flysystem\Directory;
use Move\Config;
use Symfony\Component\Yaml\Yaml;

/**
 * Class Loader
 * @package Move\Config
 */
class Loader
{
    /**
     * @var array
     */
    private $store;

    /**
     * @var Directory[]
     */
    private $directory;

    /**
     * @var string
     */
    private $defaultExtension;

    /**
     * @var callable
     */
    private $parser;

    /**
     * Filesystem constructor.
     * @param Directory $directory
     * @param string $defaultExtension
     * @param callable $parser
     */
    public function __construct(
        Directory $directory,
        $defaultExtension = '.yml',
        callable $parser = null
    ) {
        $this->directory[] = $directory;
        $this->defaultExtension = $defaultExtension;
        $this->parser = $parser ?: [Yaml::class, 'parse'];

        $this->store = [];
    }

    /**
     * @param Directory $directory
     */
    public function addFallbackDirectory(Directory $directory)
    {
        $this->directory[] = $directory;
    }

    /**
     * @param string $filename
     * @return Config|null
     * @throws \League\Flysystem\FileNotFoundException
     */
    public function load($filename)
    {
        if (isset($this->store[$filename])) {
            return $this->store[$filename];
        }

        $filename .= $this->defaultExtension;
        $config = $this->loadFile($filename, $this->directory);
        if (\is_array($config) && !empty($config)) {
            return $this->store[$filename] = new Config($config);
        }

        return $this->store[$filename] = null;
    }

    /**
     * @param string $filename
     * @param Directory[] $directories
     * @return array|false
     * @throws \League\Flysystem\FileNotFoundException
     */
    protected function loadFile(string $filename, array $directories)
    {
        foreach ($directories as $directory) {
            $config = $this->parseFile($directory, $filename);
            if ($config !== false) {
                return $config;
            }
        }
        return null;
    }

    /**
     * @param array $config
     * @param Directory $mergeDirectory
     * @param array $imports
     * @throws \League\Flysystem\FileNotFoundException
     */
    protected function mergeImports(
        array &$config,
        Directory $mergeDirectory,
        array $imports
    ) {
        foreach ($imports as $resource) {
            if (empty($resource['resource'])) {
                trigger_error('resource key must be set for import', E_USER_WARNING);
                continue;
            }

            // ajoute la directory existante
            $directories = $this->directory;
            if (!\in_array($mergeDirectory, $this->directory)) {
                array_unshift($directories, $mergeDirectory);
            }

            // charge le contenue du fichier
            $configImport = $this->loadFile($resource['resource'], $directories);
            if (\is_array($configImport)) {
                $config = array_replace_recursive($configImport, $config);
            } else {
                trigger_error('cannot load file ' . $resource['resource'] . ' for import', E_USER_WARNING);
            }
        }
    }

    /**
     * @param Directory $directory
     * @param string $filename
     * @return array|false
     * @throws \League\Flysystem\FileNotFoundException
     */
    protected function parseFile(Directory $directory, $filename)
    {
        $filepath = $directory->getPath() . '/' . $filename;
        if ($directory->getFilesystem()->has($filepath) !== false) {
            $config = \call_user_func(
                $this->parser,
                $directory->getFilesystem()->read($filepath),
                true
            );

            // si des imports sont present dans le loader :
            if (isset($config['imports']) && \is_array($config['imports'])) {
                $this->mergeImports($config, $directory, $config['imports']);
                unset($config['imports']);
            }

            return $config ?: [];
        }
        return false;
    }
}
