<?php


namespace Move\Http\Strategy\Serializer;

use League\Plates\Engine;
use League\Plates\Template\Template;
use Move\Http\Strategy\DataContainer\DataContainer;
use Move\Http\Strategy\DataContainer\EngineDataContainer;
use Psr\Http\Message\ResponseInterface;

/**
 * Class PlatesAdapterSerializer
 * @package Move\Http\Strategy\Serializer
 */
class PlatesAdapterSerializer implements SerializerInterface
{

    /**
     * @var Engine
     */
    private $engine;

    /**
     * @var callable|null
     */
    private $htmlMinifier;

    /** @var  string */
    private $templateFolderName;

    /**
     * PlatesAdapterSerializer constructor.
     * @param Engine $engine
     * @param null|string $errorFolder
     * @param callable|null $htmlMinifier
     */
    public function __construct(Engine $engine, $errorFolder = null, $htmlMinifier = null)
    {
        $this->engine = $engine;
        $this->htmlMinifier = $htmlMinifier;
        if (!empty($errorFolder)) {
            $this->engine->addFolder('error', $errorFolder);
        }
    }

    /**
     * @param string $errorFolder
     * @return $this
     */
    public function setErrorFolder($errorFolder)
    {
        try {
            $this->engine->removeFolder('error');
        } catch (\Exception $e) {
        }
        $this->engine->addFolder('error', $errorFolder);
        return $this;
    }

    /**
     * @param string $templateFolder
     * @param null|string $folderName
     * @return $this
     */
    public function setDefaultTemplateFolder($templateFolder, $folderName = null)
    {
        $this->templateFolderName = $folderName ?: '_tpl';
        try {
            $this->engine->removeFolder($this->templateFolderName);
        } catch (\Exception $e) {
        }
        $this->engine->addFolder($this->templateFolderName, $templateFolder);
        return $this;
    }

    /**
     * @param callable|null $htmlMinifier
     * @return $this
     */
    public function setHtmlMinifier($htmlMinifier)
    {
        $this->htmlMinifier = $htmlMinifier;
        return $this;
    }

    /**
     * @param mixed $dataset
     * @param ResponseInterface $response
     * @return string
     * @throws \Exception
     * @throws \Throwable
     */
    public function serialize($dataset, ResponseInterface $response)
    {
        $html = $this->render($dataset, $response);
        if (\is_callable($this->htmlMinifier)) {
            $html = \call_user_func($this->htmlMinifier, $html);
        }
        return $html;
    }

    /**
     * @param mixed $dataset
     * @param ResponseInterface $response
     * @return string
     * @throws \UnexpectedValueException
     * @throws \Throwable
     * @throws \Exception
     */
    protected function render($dataset, ResponseInterface $response)
    {
        // dans le cas d'une reponse d'erreur :
        if ($response->getStatusCode() >= 400) {
            if ($dataset instanceof DataContainer) {
                $dataset = $dataset->getDataset();
            }
            $template = $this->getErrorTemplate(
                $response->getStatusCode()
            );
            $dataset = new EngineDataContainer($template, $dataset);
        }
        if (empty($dataset['template'])) {
            throw new \UnexpectedValueException('template key must be set');
        }
        $template = $dataset['template'];
        $engineData = !empty($dataset['data']) ? $dataset['data'] : [];
        // cas ou le template est renvoyé direct
        if ($template instanceof Template) {
            return $template->render($engineData);
        }

        // applique le folder de template
        if (!empty($this->templateFolderName) && \is_string($template)) {
            $template = $this->templateFolderName . '::' . $template;
        }

        // recherche du template sinon error
        if (!$this->engine->exists($template)) {
            $template = $this->getErrorTemplate(500);
        }
        // retour
        return $this->engine->make($template)->render($engineData);
    }

    /**
     * @param string $errorStatus
     * @return string
     * @throws \UnexpectedValueException
     */
    protected function getErrorTemplate($errorStatus)
    {
        $template = 'error::' . $errorStatus;
        if (!$this->engine->exists($template)) {
            $errTemplate = 'error::40x';
            if ($errorStatus >= 500) {
                $errTemplate = 'error::50x';
            }
            if (!$this->engine->exists($errTemplate)) {
                throw new \UnexpectedValueException('error template : ' . $errTemplate . ' does not exist');
            }
            $template = $errTemplate;
        }
        return $template;
    }

    /**
     * @param mixed $dataset
     * @param ResponseInterface $response
     * @return string
     */
    public function __invoke($dataset, ResponseInterface $response)
    {
        return $this->serialize($dataset, $response);
    }
}
