<?php

namespace Cms;

use League\Flysystem\Adapter\Local;
use League\Flysystem\Directory;
use League\Flysystem\Filesystem;
use League\Flysystem\FilesystemInterface;
use Move\Config;
use Move\Config\Loader;
use Move\Environment;
use Move\Utils\Str;
use Psr\Cache\CacheItemPoolInterface;

/**
 * Class ConfigFactory
 * @package Cms
 */
class ConfigFactory
{

    /** @var FilesystemInterface */
    protected static $FILESYSTEM_HL;

    /** @var  CacheItemPoolInterface */
    protected static $CACHE_POOL;

    /** @var  string */
    protected static $CACHE_NS;

    /**
     * @param FilesystemInterface $filesystem
     */
    public static function setFileSystemHandler(FilesystemInterface $filesystem)
    {
        self::$FILESYSTEM_HL = $filesystem;
    }

    /**
     * @param CacheItemPoolInterface $cacheItemPool
     * @param string $ns
     */
    public static function setCachePool($cacheItemPool, $ns)
    {
        self::$CACHE_POOL = $cacheItemPool;
        self::$CACHE_NS = $ns;
    }

    /**
     * Renvoi le fichier de config lié a l'appName
     * @param null|string $appName
     * @param \Move\Environment|null $env
     * @return string
     */
    public static function getConfigFileName($appName = null, Environment $env = null) : string
    {
        $configFile = 'global';
        if (!empty($appName)) {
            $configFile = strtolower($appName);
        }

        $env = $env ?: new Environment();
        if ($env->isProd()) {
            $configFile .= '_prod';
        } elseif ($env->isStaging()) {
            $configFile .= '_preprod';
        }

        return $configFile;
    }

    /**
     * @param null|string $appName
     * @return Loader
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public static function getConfigLoader($appName = null) : Loader
    {
        if (!self::$FILESYSTEM_HL) {
            throw new \InvalidArgumentException('Filesystem must be defined via setFileSystemHandler');
        }

        $parserWrapper = null;
        try {
            $parserWrapper = new Config\YamlExtParserWrapper();
        } catch (\Exception $e) {
        }

        $relativePath = PathFactory::getConfigPath();
        if (self::$FILESYSTEM_HL->has($relativePath)) {
            $directory = self::$FILESYSTEM_HL->get($relativePath);
            if ($directory instanceof Directory) {
                $loader = new Loader($directory, '.yml', $parserWrapper);
            }
        }
        if ($appName) {
            $relativePath = PathFactory::getConfigPath($appName);
            $directory = self::$FILESYSTEM_HL->get($relativePath);
            if ($directory instanceof Directory) {
                if (!isset($loader)) {
                    $loader = new Loader($directory, '.yml', $parserWrapper);
                } else {
                    $loader->addFallbackDirectory($directory);
                }
            }
        }
        if (!isset($loader)) {
            throw new \RuntimeException('config not found');
        }

        // ajoute le fallback du repertoire de resources
        $resourceDir = (new Filesystem(
            new Local(__DIR__ . '/../../')
        ))->get(
            'resources/config'
        );
        if ($resourceDir instanceof Directory) {
            $loader->addFallbackDirectory($resourceDir);
        }
        return $loader;
    }

    /**
     * Configuration
     * @param null|string $appName
     * @param \Move\Environment|null $env
     * @return \Move\Config
     * @throws \Psr\Cache\InvalidArgumentException
     * @throws \InvalidArgumentException
     * @throws \League\Flysystem\FileNotFoundException
     * @throws \RuntimeException
     */
    public static function createConfig($appName = null, Environment $env = null)
    {
        if (!self::$FILESYSTEM_HL) {
            throw new \InvalidArgumentException('Filesystem must be defined via setFileSystemHandler');
        }

        // si cache existe on recherche la config en cache par le nom de fichier
        $relativePath = PathFactory::getConfigPath($appName);
        $cacheKey = Str::toAscii(self::$CACHE_NS)
            . '^' . Str::toAscii($relativePath)
            . '^' . self::$FILESYSTEM_HL->getTimestamp($relativePath);
        if (self::$CACHE_POOL && ($cacheItem = self::$CACHE_POOL->getItem($cacheKey))) {
            $cacheItem->expiresAfter(3600 * 24 * 30);
            if ($cacheItem->isHit()) {
                return $cacheItem->get();
            }
        }

        $configName = self::getConfigFileName($appName, $env);
        $configLoader = self::getConfigLoader($appName);
        $config = $configLoader->load($configName);

        if (isset($cacheItem)) {
            $cacheItem->set($config);
            self::$CACHE_POOL->save($cacheItem);
        }
        if (null === $config) {
            throw new \RuntimeException('Fichier de configuration ' . $configName . ' non chargé');
        }
        return $config;
    }
}
