<?php

namespace Cms\Serializer;

use Cms\DashParser\Entity\EntityImage;
use Cms\DashParser\Entity\EntityInterface;
use function Cms\DashParser\html_specialchars;

/**
 * Class HtmlEntityImageConverter
 * @package Cms\Serializer
 */
class HtmlEntityImageConverter extends \Cms\DashParser\Serializer\HtmlEntityImageConverter
{

    /** @var int */
    private $mediaMaxWidth;

    /**
     * HtmlEntityImageConverter constructor.
     * @param int $mediaMaxWidth
     */
    public function __construct($mediaMaxWidth = null)
    {
        $this->mediaMaxWidth = $mediaMaxWidth ?: 412;
    }

    /**
     * @param int|null $mediaMaxWidth
     */
    public function setMediaMaxWidth($mediaMaxWidth)
    {
        $this->mediaMaxWidth = $mediaMaxWidth;
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityInterface $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function getEntityNode(\DOMDocument $doc, EntityInterface $entity, $nodeText = null)
    {
        if (!$entity instanceof EntityImage) {
            return null;
        }

        $imageSource = $entity->getSrc();
        $pict = $doc->createElement('picture');
        if (is_numeric($this->mediaMaxWidth) && $this->mediaMaxWidth > 0) {
            $source = $doc->createElement('source');
            $source->setAttribute('media', '(max-width:' . $this->mediaMaxWidth . 'px)');
            $source->setAttribute('srcset', $entity->getSrc());
            $pict->appendChild($source);
        }
        $img = parent::getEntityNode($doc, $entity, $nodeText);
        $pict->appendChild($img);

        $fig = $doc->createElement('figure');
        if ($entity->getWidth()) {
            $fig->setAttribute('style', 'max-width:' . $entity->getWidth() . 'px');
        }
        $fig->setAttribute('class', 'visu');

        // parsing de la source envoyé par le cms
        if (preg_match('@^/asset/cms/(?:((?:\d+x\d+)|(?:x?\d+))/)?(\d+)(?:/config/(\d+))?$@', $imageSource, $matchs)) {
            // ajoute les donnée actuel
            if (!empty($matchs[1])) {
                $fig->setAttribute('data-current-imgsize', $matchs[1]);
            }
            if (!empty($matchs[3])) {
                $fig->setAttribute('data-current-configid', $matchs[3]);
            }
            // ajoute les donnée d'origine
            if (!empty($matchs[2])) {
                $fig->setAttribute('data-origin-assetid', $matchs[2]);
            }
        }
        $fig->appendChild($pict);

        // ajout des donnée de captions
        if (!empty($entity->getLegend()) || !empty($entity->getCopy())) {
            $figCap = $doc->createElement('figcaption');
            if (!empty($entity->getLegend())) {
                $legend = $entity->getLegend();
                html_specialchars($legend);
                $span = $doc->createElement('span', $legend);
                $span->setAttribute('class', 'desc-photo');
                $figCap->appendChild($span);
            }
            if (!empty($entity->getCopy())) {
                $copy = $entity->getCopy();
                html_specialchars($copy);
                $span = $doc->createElement('span', '© ' . $copy);
                $span->setAttribute('class', 'credits-photo');
                $figCap->appendChild($span);
            }
            $fig->appendChild($figCap);
        }

        return $fig;
    }
}
