<?php

namespace Move\Newsletter\Provider;

use Mailjet\Resources;

/**
 * Class MailjetConfirmAdapter
 * @package Move\Newsletter\Provider
 */
class MailjetConfirmAdapter extends MailjetAdapter
{

    /** @var \Move\Newsletter\Provider\MailjetNotifierAdapter */
    protected $notifier;

    /** @var array */
    private $confirmOpts;

    /** @var null|string */
    private $saltKey;

    /**
     * MailjetConfirmAdapter constructor.
     * @param string $apiKey
     * @param string $apiSecret
     * @param string $idList
     * @param array $confirmOpts
     * @param array $customProps
     */
    public function __construct(
        string $apiKey,
        string $apiSecret,
        string $idList,
        array $confirmOpts,
        array $customProps
    ) {
        parent::__construct($apiKey, $apiSecret, $idList, $customProps);
        $this->notifier = new MailjetNotifierAdapter($apiKey, $apiSecret);
        $this->confirmOpts = array_merge([
            'salt' => '',
            'fromEmail' => '',
            'fromName' => '',
            'templateId' => null,
            'templateVars' => [],
            'confirmUrl' => '',
            'subject' => '',
        ], $confirmOpts);
        $this->saltKey = $this->confirmOpts['salt'];
    }

    /**
     * @param string $confirmKey
     * @return string
     */
    public function getEmailFromConfirmKey(string $confirmKey)
    {
        $email = MailjetCryptHelper::decrypt($this->saltKey, $confirmKey);
        if (!$email) {
            throw new \InvalidArgumentException('failed to retrieve info');
        }
        return $email;
    }

    /**
     * @param string $email
     * @param array $dataset
     * @return bool
     */
    public function update(string $email, array $dataset = []) : bool
    {
        if (filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
            $email = $this->getEmailFromConfirmKey($email);
        }
        return parent::update($email, $dataset);
    }

    /**
     * load contact data into client
     * @param string $email
     * @param array $dataset
     * @return bool
     */
    private function setContactData(string $email, array $dataset) : bool
    {
        $data['Data'] = [];
        foreach ($dataset as $key => $value) {
            $data['Data'][] = ['Name' => strtolower($key), 'value' => $value];
        }
        $response = $this->client->put(Resources::$Contactdata, ['id' => $email, 'body' => $data]);
        return $this->handleResponse($response, $this->logger);
    }

    /**
     * @param string $email
     * @return bool
     */
    private function sendDoubleOpt(string $email) : bool
    {
        // make url confirm
        $urlconfirm = $this->confirmOpts['confirmUrl'] . '?key='
            . MailjetCryptHelper::encrypt($this->saltKey, $email);

        // build email body
        $builder = new MailjetEmailBodyBuilder(
            $this->confirmOpts['templateId'],
            $this->confirmOpts['subject'],
            $this->confirmOpts['fromEmail'],
            $this->confirmOpts['fromName']
        );
        $builder->addDest($email);
        $builder->setVariables(array_merge(
            ['confirmation_link' => $urlconfirm],
            $this->confirmOpts['templateVars']
        ));

        // send notification
        return $this->notifier->send($builder);
    }

    /**
     * @param string $email
     * @param array $dataset
     * @return bool
     */
    public function insert(string $email, array $dataset = []) : bool
    {
        $contactName = $dataset['name'] ?? null;
        unset($dataset['name']);
        $dataset = $this->normalizeDatas($dataset);

        // check si le contact existe
        $response = $this->client->get(Resources::$Contact, ['id' => $email]);
        if (!$response->success()) {
            // envoi du client dans la base mailjet
            $response = $this->client->post(Resources::$Contact, [
                'body' => [
                    'Email' => $email,
                    'Name' => $contactName,
                ],
            ]);
        } else {
            unset($dataset['signin_campaign'], $dataset['signin_source']);
        }
        if ($this->handleResponse($response, $this->logger)) {
            return $this->setContactData($email, $dataset)
                && $this->sendDoubleOpt($email);
        }
        return false;
    }
}
