<?php

namespace Move\Specification;

use Move\Utils\Reflection;
use PHPUnit\Framework\TestCase;

/**
 * Class PatternSpecificationTest
 * @package Move\Specification
 */
class PatternSpecificationTest extends TestCase
{
    protected $pattern = '[a-z]+';
    protected $advancedPattern = '^[0-9]+\/[1-9]+$';
    protected $delimiter = '/';
    protected $options = 'i';

    public static function providedIsSatisfiedBy()
    {
        $simple = [
            ['aaa',     true],
            ['Aaa',     true],
            ['1Aaa',    true],
            ['',        false],
            ['11',      false],
            ['!@#$%-',  false],
        ];

        $advanced = [
            ['aaa',     false],
            ['(1/2)',   false],
            ['/2',   false],
            ['0/0',    false],
            ['0/1',    true],
            ['1/2',    true],
        ];

        return [
            'simple' => $simple,
            'advanced' => $advanced
        ];
    }

    /**
     * @covers PatternSpecification::__construct
     */
    public function testConstruct()
    {
        return $this->checkConstruct($this->pattern);
    }

    /**
     * @covers PatternSpecification::__construct
     */
    public function testAdvancedConstruct()
    {
        return $this->checkConstruct($this->advancedPattern);
    }

    /**
     * @param $pattern
     * @return mixed
     */
    public function checkConstruct($pattern)
    {
        $className = PatternSpecification::class;
        $patternSpec = new $className($pattern, $this->delimiter, $this->options);
        $this->assertInstanceOf($className, $patternSpec);

        $patternValue = Reflection::getValue($patternSpec, 'pattern');
        $this->assertEquals($pattern, $patternValue);

        $delimiterValue = Reflection::getValue($patternSpec, 'delimiter');
        $this->assertEquals($this->delimiter, $delimiterValue);

        $optionsValue = Reflection::getValue($patternSpec, 'opts');
        $this->assertEquals($this->options, $optionsValue);

        return $patternSpec;
    }

    /**
     * @depends testConstruct
     * @param $patternSpec
     */
    public function testIsSatisfiedBy(PatternSpecification $patternSpec)
    {
        $this->checkIsSatisfiedBy($patternSpec,'simple');
    }

    /**
     * @depends testAdvancedConstruct
     * @param $patternSpec
     */
    public function testIsSatisfiedBy2(PatternSpecification $patternSpec)
    {
        $this->checkIsSatisfiedBy($patternSpec,'advanced');
    }

    /**
     * @param $patternSpec
     * @param $index : simple|advanced
     */
    public function checkIsSatisfiedBy(PatternSpecification $patternSpec, $index)
    {
        $datas = static::providedIsSatisfiedBy();

        foreach ($datas[$index] as $data)
        {
            $this->assertEquals($data[1], $patternSpec->isSatisfiedBy($data[0]));
        }

    }
}
