<?php

namespace Cms\Search\Campsite;

use Cms\LinkResource\LinkResourcePrefsIdentifier;
use Cms\Model\Campsite\CampsiteState;
use Cms\Model\Campsite\Feature\FeatureCode;
use Cms\Search\ElasticQueryBuilderInterface;
use Cms\Search\ElasticQueryBuilderTrait;
use Elastica\Util;
use POM\PredefinedType\GpsLocationPoint;
use function Cms\Utils\geo_diagonal;

/**
 * Class CampsiteSearchQueryBuilder
 * @package Cms\Search\Campsite
 */
class CampsiteSearchQueryBuilder implements ElasticQueryBuilderInterface
{
    use ElasticQueryBuilderTrait;

    /** Aggregat sur geo division */
    public const AGG_GEO_DIVISION = 'divisionSuggest';

    /** Aggregat sur geo address */
    public const AGG_ADDR_CITY = 'citySuggest';

    /** Aggregat sur geo address */
    public const AGG_GEO_ADMIN = 'regionSuggest';

    /** Aggregat sur les thematics */
    public const AGG_THEMATICS = 'thematicSuggest';

    /** Aggregat sur les campings ids */
    public const AGG_CAMPSITES = 'campsiteSuggest';

    /** Aggregat sur le nombre total d'emplacement */
    public const AGG_TOTAL_EMP = 'totalempSuggest';

    /** Aggregat sur le classement (etoiles) */
    public const AGG_CLASSMNT = 'classemntSuggest';

    /** Aggregat sur les camping avec piscine */
    public const AGG_POOL = 'poolMetric';

    /** Aggretat sur les camping avec une feature */
    public const AGG_COUNT_FEATURE = 'featureMetric';

    /** Aggretat sur les camping avec un produit */
    public const AGG_COUNT_PRODUCT = 'productMetric';

    /** Aggretat sur la somme des quantité d'un produit */
    public const AGG_SUM_PRODUCT = 'productSum';

    /** Aggregat sur les pays */
    public const AGG_GEO_COUNTRY = 'countrySuggest';

    /** Geo zoom (precision) */
    public const AGG_GEO_ZOOM = 'geoZoom';

    /** Geo Viewport */
    public const AGG_GEO_VIEWPORT = 'geoViewport';


    /** @var \Cms\Search\Campsite\CampsiteSearchFragmentFactory */
    private $fragmentFactory;

    /** @var string */
    private $language;

    /** @var bool */
    private $excludeDomTom;


    /**
     * CampsiteSearchQueryBuilder constructor.
     * @param int $scopeId
     * @param string $state
     * @param string $language
     * @param bool $excludeDomTom
     */
    public function __construct(
        int $scopeId,
        string $state = null,
        string $language = 'fre',
        $excludeDomTom = false
    ) {
        $this->filters['scopeId'] = $scopeId;
        $this->filters['state'] = $state;
        $this->filters['codeConcat'] = null;
        $this->filters['codeParent'] = null;
        $this->filters['cityId'] = null;
        $this->filters['cityName'] = null;
        $this->filters['excludeCityId'] = null;
        $this->filters['features'] = null;
        $this->filters['campsiteIds'] = null;
        $this->filters['campsiteIdsWithTh'] = null;
        $this->filters['thematicIds'] = null;
        $this->filters['gpsLocation'] = null;
        $this->filters['excludeCampsiteIds'] = null;
        $this->filters['countryCode'] = null;
        $this->filters['additional'] = [];
        $this->filters['types'] = null;
        $this->filters['slug_start'] = null;
        $this->filters['codePostal'] = null;
        $this->filters['modified_at'] = null;
        $this->language = $language;
        $this->fragmentFactory = new CampsiteSearchFragmentFactory();
        $this->excludeDomTom = $excludeDomTom;
    }

    /**
     * @param bool $excludeDomTom
     * @return CampsiteSearchQueryBuilder
     */
    public function excludeDomTom(bool $excludeDomTom) : CampsiteSearchQueryBuilder
    {
        $this->excludeDomTom = $excludeDomTom;
        return $this;
    }

    /**
     * @param string $slug
     * @return $this
     */
    public function setSlugStart(string $slug)
    {
        $this->filters['slug_start'] = $slug;
        return $this;
    }

    /**
     * @param array $types
     * @return $this
     */
    public function setCampsiteTypes(array $types)
    {
        $this->filters['types'] = $types;
        return $this;
    }

    /**
     * @param array $types
     * @param null|array $features
     * @return $this
     */
    public function setServiceAreas(array $types, array $features = null)
    {
        $this->filters['service_area'][] = [$types, $features];
        return $this;
    }

    /**
     * @param string $postalCode
     * @return $this
     */
    public function setPostalCode(string $postalCode)
    {
        $this->filters['codePostal'] = $postalCode;
        return $this;
    }

    /**
     * @param array $filters
     * @param bool $merge
     * @return $this
     */
    public function setAdditionalFilters(array $filters, bool $merge = true)
    {
        if (!$merge) {
            $this->filters['additional'] = $filters;
        } else {
            $this->filters['additional'] = array_merge(
                $this->filters['additional'],
                $filters
            );
        }

        return $this;
    }

    /**
     * @param array $thematicIds
     * @param array $campsiteIds
     * @return $this
     */
    public function setCampsiteWithThematic(array $thematicIds, array $campsiteIds)
    {
        $this->setThematics($thematicIds);
        $this->filters['campsiteIdsWithTh'] = $campsiteIds;
        return $this;
    }

    /**
     * @param array|string $codes
     * @return $this
     */
    public function setCountryCode($codes)
    {
        $this->filters['countryCode'] = (array)$codes;
        return $this;
    }

    /**
     * @param int $scopeId
     * @return $this
     */
    public function setScopeId(int $scopeId)
    {
        $this->filters['scopeId'] = $scopeId;
        return $this;
    }

    /**
     * @param array $thematicIds
     * @return $this
     */
    public function setThematics(array $thematicIds)
    {
        $this->filters['thematicIds'] = $thematicIds;
        return $this;
    }

    /**
     * @param string $state
     * @return $this
     */
    public function setState(string $state = null)
    {
        $this->filters['state'] = $state;
        return $this;
    }

    /**
     * @param array $featureCodes
     * @param float|null $boost
     * @return $this
     */
    public function hasFeatures(array $featureCodes, float $boost = null)
    {
        $this->filters['features'] = [$boost, $featureCodes];
        return $this;
    }

    /**
     * @param string $codeParent
     * @return $this
     */
    public function setAdminCodeParent(string $codeParent = null)
    {
        $this->filters['codeParent'] = $codeParent;
        return $this;
    }

    /**
     * @param array|string $codeConcat
     * @return $this
     */
    public function setAdminCodeConcat($codeConcat = null)
    {
        $this->filters['codeConcat'] = (array)$codeConcat;
        return $this;
    }

    /**
     * @param \DateTime $modifiedBegin
     * @param \DateTime $modifiedEnd
     * @return $this
     */
    public function setModifiedAtRange(\DateTime $modifiedBegin = null, \DateTime $modifiedEnd = null)
    {
        $this->filters['modified_at'] = [$modifiedBegin, $modifiedEnd];
        return $this;
    }

    /**
     * @param \POM\PredefinedType\GpsLocationPoint $gpsLocation
     * @param string $distance
     * @return $this
     */
    public function setGpsLocation(GpsLocationPoint $gpsLocation, string $distance)
    {
        $this->filters['gpsLocation'] = [$gpsLocation, $distance];
        return $this;
    }

    /**
     * @param int $geodataCityId
     * @return $this
     */
    public function excludeCityGeodataId(int $geodataCityId = null)
    {
        $this->filters['excludeCityId'] = $geodataCityId;
        return $this;
    }

    /**
     * @param int $geodataCityId
     * @return $this
     */
    public function setCityGeodataId(int $geodataCityId = null)
    {
        $this->filters['cityId'] = $geodataCityId;
        return $this;
    }

    /**
     * @param string $geoSearch
     * @return $this
     */
    public function setGeoSearch(string $geoSearch)
    {
        $this->filters['geoSearch'] = $geoSearch;
        return $this;
    }

    /**
     * @param array $campsiteIds
     * @param float|null $boost
     * @return $this
     */
    public function setCampsiteId(array $campsiteIds, float $boost = null)
    {
        if (!empty($campsiteIds)) {
            $this->filters['campsiteIds'] = [$boost, $campsiteIds];
        }
        return $this;
    }

    /**
     * @param array $northEast
     * @param array $southWest
     * @return $this
     */
    public function setBoundingBox(array $northEast, array $southWest)
    {
        // filter bounding
        $boundingFilter = $this->getFragmentFactory()->getAddressBoundingBoxFilter(
            array_reverse($northEast), // inverse north_east GMaps
            array_reverse($southWest), // inverse south_west GMaps
            LinkResourcePrefsIdentifier::ADDRESS_LOC
        );
        $this->setAdditionalFilters([
            $this->getFragmentFactory()->getNestedQuery(
                $this->getFragmentFactory()->address_nested_path,
                $boundingFilter
            ),
        ]);
        return $this;
    }

    /**
     * @param array $northEast
     * @param array $southWest
     * @param int|null $precision
     * @param bool $addInFilter
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function setBoundingBoxFacet(
        array $northEast,
        array $southWest,
        int &$precision = null,
        bool $addInFilter = true
    ) {
        // distance entre les deux points pour trouver la bonne precision
        $diagonal = geo_diagonal($northEast, $southWest);

        /*
            3 > 156.5km x 156km
            4 > 39.1km x 19.5km
            5 > 4.9km x 4.9km
            6 > 1.2km x 609.4m
        */
        if ($precision === null) {
            $precision = 2;
            if ($diagonal < 50) {
                $precision = null;
            } elseif ($diagonal < 600) {
                $precision = 4;
            } elseif ($diagonal < 1500) {
                $precision = 3;
            }
        }

        // filter bounding
        $boundingFilter = $this->getFragmentFactory()->getAddressBoundingBoxFilter(
            array_reverse($northEast), // inverse north_east GMaps
            array_reverse($southWest), // inverse south_west GMaps
            LinkResourcePrefsIdentifier::ADDRESS_LOC
        );

        // finalisation du builder (on prepare les deux pour le zoom automatique si besoin)
        $this->addFacet(
            static::AGG_GEO_ZOOM,
            [$precision],
            [$boundingFilter]
        );
        if ($addInFilter === true) {
            $this->setAdditionalFilters([
                $this->getFragmentFactory()->getNestedQuery(
                    $this->getFragmentFactory()->address_nested_path,
                    $boundingFilter
                ),
            ]);
        }
        return $this;
    }

    /**
     * @param array $campsiteIds
     * @return $this
     */
    public function excludeCampsiteIds(array $campsiteIds)
    {
        $this->filters['excludeCampsiteIds'] = $campsiteIds;
        return $this;
    }

    /**
     * @param array $featureCodeMap
     * @param array $featureCodes
     * @param array $facetFitlers
     * @return $this
     */
    public function addFeatureFacet(
        array $featureCodeMap,
        array $featureCodes = [],
        array $facetFitlers = []
    ) {
        $facetName = static::AGG_COUNT_FEATURE;
        $this->hasAggregate($facetName, $featureCodeMap);

        $aggFilters = [];
        $aggFilter['bool'] = [
            'must' => array_values(array_filter($facetFitlers)),
        ];
        $featureCodeSelected = array_intersect_key(
            $featureCodeMap,
            array_fill_keys($featureCodes, null)
        );
        if (!empty($aggFilter['bool']['must']) || $featureCodeSelected) {
            foreach ($featureCodeMap as $featureName => $featureCode) {
                $aggFilters[$featureName] = !empty($aggFilter['bool']['must']) ? $aggFilter : [];
                foreach ($featureCodeSelected as $name => $code) {
                    if ($name === $featureName) {
                        continue;
                    }
                    $aggFilters[$featureName]['bool']['must'][] = $this->getFragmentFactory()
                        ->getFeatureCodeSearchQuery([$code]);
                }
            }
            $this->hasAggregateFilters($facetName, $aggFilters);
        }
        return $this;
    }

    /**
     * @param string|null $queryString
     * @param string $addressType
     * @return array
     * @deprecated
     */
    public function getAddressBoolSearchQuery(string $queryString = null, string $addressType) : array
    {
        return $this->fragmentFactory->getAddressBoolSearchQuery($queryString, $addressType);
    }

    /**
     * @return array
     */
    public function getAggsAsArray() : array
    {
        if (!$this->aggs) {
            return [];
        }

        // init aggs
        $aggsQuery = [];

        // aggregat des campsiteIds
        if (isset($this->aggs[self::AGG_CAMPSITES])) {
            $aggFragment = [
                'terms' => [
                    'field' => 'id',
                    'size' => 10000,
                ],
            ];
            if (!empty($this->aggsFilter[self::AGG_CAMPSITES])) {
                $aggsQuery[self::AGG_CAMPSITES . '_filter'] = [
                    'filter' => $this->aggsFilter[self::AGG_CAMPSITES],
                    'aggs' => [
                        self::AGG_CAMPSITES => $aggFragment,
                    ],
                ];
            } else {
                $aggsQuery[self::AGG_CAMPSITES] = $aggFragment;
            }
        }

        // recherche uniquement sur les thematic
        if (isset($this->aggs[self::AGG_THEMATICS])) {
            $aggFragment = [
                'terms' => [
                    'size' => 100,
                    'field' => 'campsite_thematics',
                    'order' => ['_count' => 'desc'],
                ],
                'aggs' => [
                    'campsites' => [
                        'terms' => [
                            'size' => 10000,
                            'field' => 'id',
                        ],
                    ],
                ],
            ];
            if (!empty($this->aggsFilter[self::AGG_THEMATICS])) {
                $aggsQuery[self::AGG_THEMATICS . '_filter'] = [
                    'filter' => $this->aggsFilter[self::AGG_THEMATICS],
                    'aggs' => [
                        self::AGG_THEMATICS => $aggFragment,
                    ],
                ];
            } else {
                $aggsQuery[self::AGG_THEMATICS] = $aggFragment;
            }
        }

        // sum des quantité de produit
        if (!empty($this->aggs[self::AGG_SUM_PRODUCT])
            && !empty($this->aggsFilter[self::AGG_SUM_PRODUCT])
        ) {
            $aggFragment = [
                'terms' => [
                    'size' => 10000,
                    'field' => 'id',
                ],
                'aggs' => [
                    self::AGG_SUM_PRODUCT . '_nested' => [
                        'nested' => [
                            'path' => $this->fragmentFactory->product_nested_path,
                        ],
                        'aggs' => [
                            self::AGG_SUM_PRODUCT . '_filter' => [
                                'filter' => $this->aggsFilter[self::AGG_SUM_PRODUCT],
                                'aggs' => [
                                    self::AGG_SUM_PRODUCT . '_sum' => [
                                        'sum' => [
                                            'field' => $this->fragmentFactory->product_nested_path . '.quantity',
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                    self::AGG_SUM_PRODUCT => [
                        'bucket_selector' => [
                            'buckets_path' => [
                                'totalLoc' => self::AGG_SUM_PRODUCT . '_nested>'
                                    . self::AGG_SUM_PRODUCT . '_filter>'
                                    . self::AGG_SUM_PRODUCT . '_sum',
                            ],
                            'script' => 'params.totalLoc ' . $this->aggs[self::AGG_SUM_PRODUCT],
                        ],
                    ],
                ],
            ];
            $aggsQuery[self::AGG_SUM_PRODUCT] = $aggFragment;
            /*$aggsQuery[self::AGG_SUM_PRODUCT . '_stats'] = [
                'stats_bucket' => [
                    'buckets_path' => self::AGG_SUM_PRODUCT . '>'
                        . self::AGG_SUM_PRODUCT . '_nested>'
                        . self::AGG_SUM_PRODUCT . '_filter>'
                        . self::AGG_SUM_PRODUCT . '_sum',
                ],
            ];*/
        }

        // recherche uniquement sur un produit
        if (!empty($this->aggs[self::AGG_COUNT_PRODUCT])
            && \is_array($this->aggs[self::AGG_COUNT_PRODUCT])
        ) {
            $searchQueryCodes = [];
            foreach ($this->aggs[self::AGG_COUNT_PRODUCT] as $productId) {
                $searchQueryCodes['product_id#' . $productId] = $this->getTermQuery(
                    'campsite_products.id',
                    $productId
                );
            }
            $aggFragment = [
                'filters' => ['filters' => $searchQueryCodes],
            ];
            if (!empty($this->aggsFilter[self::AGG_COUNT_PRODUCT])) {
                $aggsQuery[self::AGG_COUNT_PRODUCT . '_filter'] = [
                    'filter' => $this->aggsFilter[self::AGG_COUNT_PRODUCT],
                    'aggs' => [
                        self::AGG_COUNT_PRODUCT => $aggFragment,
                    ],
                ];
            } else {
                $aggsQuery[self::AGG_COUNT_PRODUCT] = $aggFragment;
            }
        }

        // filtres sur les features
        if (isset($this->aggs[self::AGG_CLASSMNT])
            || isset($this->aggs[self::AGG_POOL])
            || !empty($this->aggs[self::AGG_COUNT_FEATURE])
        ) {
            // liste des aggregats inferieur
            $subAggs = [];
            $nestedPath = $this->fragmentFactory->feature_nested_path;

            // recherche uniquement sur une feature
            if (!empty($this->aggs[self::AGG_COUNT_FEATURE])
                && \is_array($this->aggs[self::AGG_COUNT_FEATURE])
            ) {
                $aggFragment = [];
                foreach ($this->aggs[self::AGG_COUNT_FEATURE] as $featureName => $featureCode) {
                    $filter['bool'] = [
                        'must' => [
                            $this->getNestedQuery(
                                $nestedPath,
                                $this->getTermsQuery(
                                    $nestedPath . '.feature_code',
                                    (array)$featureCode
                                )
                            ),
                        ],
                    ];
                    if (!empty($this->aggsFilter[self::AGG_COUNT_FEATURE][$featureName])) {
                        $filter['bool']['must'][] = $this->aggsFilter[self::AGG_COUNT_FEATURE][$featureName];
                    }
                    $aggFragment[$featureName . '_filter'] = [
                        'filter' => $filter,
                        'aggs' => [
                            $featureName => [
                                'value_count' => ['field' => 'id'],
                            ],
                        ],
                    ];
                }
                $aggsQuery = array_merge($aggsQuery, $aggFragment);
            }

            // recherche uniquement sur les piscines
            if (isset($this->aggs[self::AGG_POOL])) {
                $aggFragment = [
                    'filter' => $this->fragmentFactory->getPoolSearchQuery(),
                    'aggs' => [
                        self::AGG_POOL . '_to_campsite' => [
                            'reverse_nested' => new \stdClass(),
                            'aggs' => [
                                self::AGG_POOL => [
                                    'value_count' => ['field' => 'id'],
                                ],
                            ],
                        ],
                    ],
                ];
                if (!empty($this->aggsFilter[self::AGG_POOL])) {
                    $aggsQuery[self::AGG_POOL . '_filter'] = [
                        'filter' => $this->aggsFilter[self::AGG_POOL],
                        'aggs' => [
                            self::AGG_POOL . '_nested' => [
                                'nested' => [
                                    'path' => $nestedPath,
                                ],
                                'aggs' => [
                                    self::AGG_POOL . '_filter' => $aggFragment,
                                ],
                            ],
                        ],
                    ];
                } else {
                    $subAggs[self::AGG_POOL . '_filter'] = $aggFragment;
                }
            }

            // recherche uniquement sur une feature
            if (isset($this->aggs[self::AGG_CLASSMNT])) {
                $searchQuery = $this->getTermQuery(
                    $nestedPath . '.feature_code',
                    FeatureCode::CLASSEMENT
                );
                $aggFragment = [
                    'filter' => $searchQuery,
                    'aggs' => [
                        self::AGG_CLASSMNT => [
                            'terms' => [
                                'size' => $this->aggsSize,
                                'field' => $nestedPath . '.value',
                                'order' => ['_count' => 'desc'],
                            ],
                        ],
                    ],
                ];
                if (!empty($this->aggsFilter[self::AGG_CLASSMNT])) {
                    $aggsQuery['clssmnt_filter'] = [
                        'filter' => $this->aggsFilter[self::AGG_CLASSMNT],
                        'aggs' => [
                            'clssmnt_nested' => [
                                'nested' => [
                                    'path' => $nestedPath,
                                ],
                                'aggs' => ['clssmnt' => $aggFragment],
                            ],
                        ],
                    ];
                } else {
                    $subAggs['clssmnt'] = $aggFragment;
                }
            }

            if (!empty($subAggs)) {
                $aggsQuery['features'] = [
                    'nested' => [
                        'path' => $nestedPath,
                    ],
                    'aggs' => $subAggs,
                ];
            }
        }

        // recherche uniquement le nombre total d'emplacement
        if (isset($this->aggs[self::AGG_TOTAL_EMP])) {
            $nestedPath = 'features_nested';
            $searchQuery = $this->getTermQuery(
                $nestedPath . '.feature_code',
                FeatureCode::NBTOTALEMP
            );
            $aggFragment = $this->getAggsNestedQuery(
                $nestedPath,
                $searchQuery,
                [
                    self::AGG_TOTAL_EMP => [
                        'range' => [
                            'field' => $nestedPath . '.quantity',
                            'ranges' => [
                                ['to' => 100],
                                ['from' => 100, 'to' => 300],
                                ['from' => 300],
                            ],
                        ],
                    ],
                ]
            );
            if (!empty($this->aggsFilter[self::AGG_TOTAL_EMP])) {
                $aggsQuery['nbtotalemp_filter'] = [
                    'filter' => $this->aggsFilter[self::AGG_TOTAL_EMP],
                    'aggs' => [
                        'nbtotalemp' => $aggFragment,
                    ],
                ];
            } else {
                $aggsQuery['nbtotalemp'] = $aggFragment;
            }
        }

        // recherche uniquement sur la division geographique
        if (isset($this->aggs[self::AGG_GEO_ADMIN])) {
            $searchQuery = ['match_all' => (object)[]];
            $nestedPath = $this->fragmentFactory->admin_nested_path;
            if (!empty($this->aggs[self::AGG_GEO_ADMIN])
                && !\is_bool($this->aggs[self::AGG_GEO_ADMIN])
            ) {
                $searchQuery = $this->getTermQuery(
                    $nestedPath . '.admin_level',
                    $this->aggs[self::AGG_GEO_ADMIN]
                );
            } elseif (!empty($this->aggsFilter[self::AGG_GEO_ADMIN])) {
                $searchQuery = $this->aggsFilter[self::AGG_GEO_ADMIN];
            }
            $aggsQuery['geo_admins'] = $this->getAggsNestedQuery(
                $nestedPath,
                $searchQuery,
                [
                    self::AGG_GEO_ADMIN => [
                        'terms' => [
                            'size' => 300,
                            'field' => $nestedPath . '.id',
                            'order' => ['_count' => 'desc'],
                        ],
                    ],
                    self::AGG_GEO_ADMIN . '_parent' => [
                        'terms' => [
                            'size' => 300,
                            'field' => $nestedPath . '.parent_id',
                            'order' => ['_count' => 'desc'],
                        ],
                    ],
                ]
            );
        }

        // recherche uniquement sur la division geographique
        if (isset($this->aggs[self::AGG_GEO_DIVISION])) {
            $searchQuery = ['match_all' => (object)[]];
            if (!empty($this->aggs[self::AGG_GEO_DIVISION])) {
                $searchQuery = $this->fragmentFactory->getDivisionSearchQuery(
                    $this->language,
                    (string)$this->aggs[self::AGG_GEO_DIVISION]
                );
            }
            $nestedPath = 'geo_divisions.division_name_i18n';
            $aggsQuery['divisions'] = $this->getAggsNestedQuery(
                $nestedPath,
                $searchQuery,
                [
                    self::AGG_GEO_DIVISION => [
                        'terms' => [
                            'size' => 300,
                            'field' => $nestedPath . '.id',
                            'order' => ['_count' => 'desc'],
                        ],
                    ],
                ]
            );
        }

        // recherche uniquement sur l'address geographique
        if (isset($this->aggs[self::AGG_ADDR_CITY])) {
            $searchQuery = $this->fragmentFactory->getAddressBoolSearchQuery(
                null,
                LinkResourcePrefsIdentifier::ADDRESS_LOC
            );
            if (!empty($this->aggs[self::AGG_ADDR_CITY])
                && !\is_bool($this->aggs[self::AGG_ADDR_CITY])
            ) {
                $searchQuery = $this->fragmentFactory->getAddressBoolSearchQuery(
                    (string)$this->aggs[self::AGG_ADDR_CITY],
                    LinkResourcePrefsIdentifier::ADDRESS_LOC
                );
            }
            $nestedPath = $this->fragmentFactory->address_nested_path;
            $aggsQuery['addresses'] = $this->getAggsNestedQuery(
                $nestedPath,
                $searchQuery,
                [
                    self::AGG_ADDR_CITY => [
                        'terms' => [
                            'size' => 40000,
                            'field' => $nestedPath . '.city_geodata_id',
                            'order' => ['_count' => 'desc'],
                        ],
                    ],
                ]
            );
        }

        // recherche uniquement sur l'address geographique
        if (isset($this->aggs[self::AGG_GEO_COUNTRY])) {
            $searchQuery = $this->fragmentFactory->getAddressBoolSearchQuery(
                null,
                LinkResourcePrefsIdentifier::ADDRESS_LOC
            );
            $nestedPath = $this->fragmentFactory->address_nested_path;
            $aggsQuery['addressesCountry'] = $this->getAggsNestedQuery(
                $nestedPath,
                $searchQuery,
                [
                    self::AGG_GEO_COUNTRY => [
                        'terms' => [
                            'size' => 210,
                            'field' => $nestedPath . '.country_code',
                            'order' => ['_count' => 'desc'],
                        ],
                    ],
                ]
            );
        }

        // zoom geo
        if (!empty($this->aggs[self::AGG_GEO_ZOOM])) {
            $nestedPath = $this->fragmentFactory->address_nested_path;
            $aggFragment = [
                'geohash_grid' => [
                    'size' => 80,
                    'precision' => $this->aggs[self::AGG_GEO_ZOOM],
                    'field' => $nestedPath . '.geo_location',
                ],
                'aggs' => [
                    'centroid_' . self::AGG_GEO_ZOOM => [
                        'geo_centroid' => [
                            'field' => $nestedPath . '.geo_location',
                        ],
                    ],
                ],
            ];

            $this->appendAggFragment($aggsQuery, self::AGG_GEO_ZOOM, $aggFragment, $nestedPath);
        }

        if (!empty($this->aggs[self::AGG_GEO_VIEWPORT])) {
            $nestedPath = $this->fragmentFactory->address_nested_path;
            $aggFragment = [
                'geo_bounds' => [
                    'field' => $nestedPath . '.geo_location',
                ],
            ];

            $this->appendAggFragment($aggsQuery, self::AGG_GEO_VIEWPORT, $aggFragment, $nestedPath);
        }

        return $aggsQuery;
    }

    /**
     * @return array
     */
    public function getQueryAsArray() : array
    {
        // filtre obligatoire
        $queryParams = $this->query_params;
        $queryParams['bool']['filter'][] = [
            'term' => ['scope_id' => $this->filters['scopeId']],
        ];
        if ($this->filters['state']) {
            $queryParams['bool']['filter'][] = [
                'term' => ['state' => $this->filters['state']],
            ];
            if ($this->filters['state'] === CampsiteState::ONLINE) {
                $queryParams['bool']['filter'][] = [
                    'bool' => [
                        'must_not' => [
                            [
                                'term' => [
                                    'campsite_accept_vac' => false,
                                ],
                            ],
                        ],
                    ],
                ];
            }
        }
        if ($this->filters['types']) {
            $queryParams['bool']['filter'][] = [
                'terms' => ['type' => $this->filters['types']],
            ];
        }
        if ($this->filters['slug_start']) {
            $queryParams['bool']['must'][] = [
                'prefix' => [
                    'slug' => [
                        'value' => $this->filters['slug_start'],
                    ],
                ],
            ];
        }

        // ajout du filtre sur les features
        if ($this->filters['features']) {
            [$featureBoost, $features] = $this->filters['features'];
            $termsQuery = $this->getTermsQuery(
                'features.feature_code',
                $features,
                $featureBoost
            );
            if (!$featureBoost) {
                $queryParams['bool']['filter'][] = $termsQuery;
            } elseif ($featureBoost > 0) {
                $queryParams['bool']['should'][] = $termsQuery;
            } else {
                $queryParams['bool']['must_not'][] = $termsQuery;
            }
        }

        // ajout du filtre sur les id de camping avec thematic
        if ($this->filters['campsiteIdsWithTh']) {
            $boolQuery[] = $this->getTermsQuery(
                '_id',
                $this->filters['campsiteIdsWithTh']
            );
            // ajout du filtre sur les thematics
            if ($this->filters['thematicIds']) {
                $termsQuery = $this->getTermsQuery(
                    'campsite_thematics',
                    $this->filters['thematicIds']
                );
                $boolQuery[] = $termsQuery;
            }
            $queryParams['bool']['must'][] = $this->getBoolQuery($boolQuery, 'should');
        } // ajout du filtre sur les thematics
        elseif ($this->filters['thematicIds']) {
            $termsQuery = $this->getTermsQuery(
                'campsite_thematics',
                $this->filters['thematicIds']
            );
            $queryParams['bool']['filter'][] = $termsQuery;
        }

        // ajoute les filtre additionel
        if ($this->filters['additional']) {
            foreach ($this->filters['additional'] as $additionalFilter) {
                $queryParams['bool']['filter'][] = $additionalFilter;
            }
        }

        // recherche par date
        if ($this->filters['modified_at']) {
            if ($this->filters['modified_at'][0] instanceof \DateTime) {
                if ($this->filters['modified_at'][1] instanceof \DateTime) {
                    $queryParams['bool']['filter'][] = $this->getRangeQuery(
                        'modified_at',
                        $this->filters['modified_at'][1]->format('Y-m-d'),
                        $this->filters['modified_at'][0]->format('Y-m-d')
                    );
                } else {
                    $queryParams['bool']['filter'][] = $this->getRangeQuery(
                        'modified_at',
                        null,
                        $this->filters['modified_at'][0]->format('Y-m-d')
                    );
                }
            } elseif ($this->filters['modified_at'][1] instanceof \DateTime) {
                $queryParams['bool']['filter'][] = $this->getRangeQuery(
                    'modified_at',
                    $this->filters['modified_at'][1]->format('Y-m-d')
                );
            }
        }

        // ajout du filtre sur les id de camping
        if ($this->filters['campsiteIds']) {
            [$campsiteBoost, $ids] = $this->filters['campsiteIds'];
            if (!$campsiteBoost) {
                $queryParams['bool']['filter'][] = $this->getTermsQuery('_id', $ids);
            } else {
                $queryParams['bool']['filter'][] = $this->getTermsQuery('_id', $ids, $campsiteBoost);
            }
        }

        // ajout du filtre d'exclusion sur les id de camping
        if ($this->filters['excludeCampsiteIds']) {
            $queryParams['bool']['must'][] = $this->getBoolQuery(
                $this->getTermsQuery('_id', $this->filters['excludeCampsiteIds']),
                'must_not'
            );
        }

        // recherche par ville
        if (!empty($this->filters['geoSearch'])) {
            $stringQuery = [];
            // recherche uniquement sur l'address geographique
            $nestedQuery = $this->getNestedQuery(
                $this->fragmentFactory->address_nested_path,
                [
                    $this->fragmentFactory->getAddressBoolSearchQuery(
                        $this->filters['geoSearch'],
                        LinkResourcePrefsIdentifier::ADDRESS_LOC
                    ),
                ]
            );
            $stringQuery[] = $nestedQuery;

            // recherche uniquement sur geo_admin
            $nestedQuery = $this->getNestedQuery(
                $this->fragmentFactory->admin_i18n_nested_path,
                [
                    $this->fragmentFactory->getAdminSearchQuery(
                        $this->language,
                        $this->filters['geoSearch']
                    ),
                ]
            );
            $stringQuery[] = $nestedQuery;

            $queryParams['bool']['must'][] = $this->getBoolQuery($stringQuery, 'should');
        }

        // lecture de la requete
        $queryString = Util::escapeTerm(
            $this->filters['searchText'] ?: ''
        );
        if (!empty($queryString)) {
            $stringQuery = [];
            // recherche sur les champs texte suivant
            $stringQuery[] = [
                'match' => [
                    'campsite_name' => [
                        'query' => $queryString,
                        'boost' => 8,
                        //'operator' => 'and',
                    ],
                ],
            ];

            // recherche uniquement sur l'address geographique
            $nestedQuery = $this->getNestedQuery(
                $this->fragmentFactory->address_nested_path,
                [
                    $this->fragmentFactory->getAddressBoolSearchQuery(
                        $queryString,
                        LinkResourcePrefsIdentifier::ADDRESS_LOC
                    ),
                ]
            );
            $stringQuery[] = $nestedQuery;
            unset($subQuery);

            // recherche uniquement sur geo_division
            $nestedQuery = $this->getNestedQuery(
                $this->fragmentFactory->division_nested_path,
                [$this->fragmentFactory->getDivisionSearchQuery($this->language, $queryString)]
            );
            $stringQuery[] = $nestedQuery;
            unset($subQuery);

            $queryParams['bool']['must'][] = $this->getBoolQuery($stringQuery, 'should');
        }

        // exclusion des ville
        if (($excludeCityId = $this->filters['excludeCityId']) !== null) {
            $nestedPath = $this->fragmentFactory->address_nested_path;
            $subQuery = $this->getBoolQuery([
                $this->getBoolQuery([
                    $this->getTermQuery($nestedPath . '.city_geodata_id', $excludeCityId),
                ], 'must_not'),
                $this->getTermQuery(
                    $nestedPath . '.type',
                    LinkResourcePrefsIdentifier::ADDRESS_LOC
                ),
            ]);
            $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
            $queryParams['bool']['filter'][] = $nestedQuery;
        }

        // filtre sur la zone gps
        if ($this->filters['gpsLocation']) {
            [$gpsLocation, $distance] = $this->filters['gpsLocation'];
            if ($gpsLocation instanceof GpsLocationPoint) {
                $nestedPath = $this->fragmentFactory->address_nested_path;
                $subQuery = $this->getBoolQuery([
                    [
                        'geo_distance' => [
                            'distance' => $distance,
                            $nestedPath . '.geo_location' => [
                                'lat' => $gpsLocation->latitude,
                                'lon' => $gpsLocation->longitude,
                            ],
                        ],
                    ],
                    $this->getTermQuery(
                        $nestedPath . '.type',
                        LinkResourcePrefsIdentifier::ADDRESS_LOC
                    ),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery], true);
                $queryParams['bool']['filter'][] = $nestedQuery;
                // filtre sur la ville
                if (($cityId = $this->filters['cityId']) !== null) {
                    $nestedPath = $this->fragmentFactory->address_nested_path;
                    $subQuery = $this->getBoolQuery([
                        $this->getTermQuery($nestedPath . '.city_geodata_id', $cityId),
                        $this->getTermQuery(
                            $nestedPath . '.type',
                            LinkResourcePrefsIdentifier::ADDRESS_LOC
                        ),
                    ]);
                    $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                    $queryParams['bool']['must_not'][] = $nestedQuery;
                }
                // filtre sur le pays
                if ($this->filters['countryCode']) {
                    $queryParams['bool']['filter'][] = $this->fragmentFactory->getAddressCountryQuery(
                        $this->filters['countryCode'],
                        LinkResourcePrefsIdentifier::ADDRESS_LOC,
                        $this->excludeDomTom
                    );
                }
            }
        } else {
            // filtre sur la ville
            if (($cityId = $this->filters['cityId']) !== null) {
                $nestedPath = $this->fragmentFactory->address_nested_path;
                $subQuery = $this->getBoolQuery([
                    $this->getTermQuery($nestedPath . '.city_geodata_id', $cityId),
                    $this->getTermQuery(
                        $nestedPath . '.type',
                        LinkResourcePrefsIdentifier::ADDRESS_LOC
                    ),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                $queryParams['bool']['filter'][] = $nestedQuery;
            }
            if (($postalCode = $this->filters['codePostal']) !== null) {
                $nestedPath = $this->fragmentFactory->address_nested_path;
                $subQuery = $this->getBoolQuery([
                    $this->getTermQuery($nestedPath . '.postal_code', $postalCode),
                    $this->getTermQuery(
                        $nestedPath . '.type',
                        LinkResourcePrefsIdentifier::ADDRESS_LOC
                    ),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                $queryParams['bool']['filter'][] = $nestedQuery;
            }
            // filtre sur la zone administrative
            if (($adminCodeConcat = $this->filters['codeConcat']) !== null) {
                $nestedPath = $this->fragmentFactory->admin_nested_path;
                $subQuery = $this->getBoolQuery([
                    $this->getTermsQuery($nestedPath . '.admin_code_concat', $adminCodeConcat),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                $queryParams['bool']['filter'][] = $nestedQuery;
            }
            // filtre sur la zone administrative
            if (($adminCodeParent = $this->filters['codeParent']) !== null) {
                $nestedPath = $this->fragmentFactory->admin_nested_path;
                $subQuery = $this->getBoolQuery([
                    $this->getTermQuery($nestedPath . '.admin_code_parent', $adminCodeParent),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                $queryParams['bool']['filter'][] = $nestedQuery;
            }
            // filtre sur le pays
            if (!$cityId
                && !$adminCodeConcat
                && !$adminCodeParent
                && $this->filters['countryCode']
            ) {
                $queryParams['bool']['filter'][] = $this->fragmentFactory->getAddressCountryQuery(
                    $this->filters['countryCode'],
                    LinkResourcePrefsIdentifier::ADDRESS_LOC,
                    $this->excludeDomTom
                );
            }
        }

        $query = [
            'query' => $this->applyFunctionScore($queryParams),
        ];
        return $query;
    }

    /**
     * @return \Cms\Search\Campsite\CampsiteSearchFragmentFactory
     */
    public function getFragmentFactory() : CampsiteSearchFragmentFactory
    {
        return $this->fragmentFactory;
    }
}
