<?php

namespace Cms\Worker\Scheduler;

use Cms\Client\Scheduler\Model\TaskRestfulModel;
use Cms\Client\Scheduler\SchedulerTaskClient;
use Cms\Model\Scheduler\LinkResource\LinkResourceModel;
use Cms\Worker\WorkerCommandFactory;
use Move\Command\CommandBusInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Class AbstractSchedulerCommandHandler
 * @package Cms\Worker\Scheduler
 */
abstract class AbstractSchedulerCommandHandler implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    /** @var  SchedulerTaskClient */
    protected $taskClient;

    /** @var CommandBusInterface */
    protected $commandBus;

    /** @var resource */
    private $logStream;

    /**
     * SchedulerCommandHandler constructor.
     * @param SchedulerTaskClient $taskClient
     * @param CommandBusInterface $taskBus
     * @param resource $logStream
     * @throws \InvalidArgumentException
     */
    public function __construct(
        SchedulerTaskClient $taskClient,
        CommandBusInterface $taskBus,
        $logStream
    ) {
        if (!\is_resource($logStream)) {
            throw new \InvalidArgumentException('stream must be a valid resource');
        }
        $this->logStream = $logStream;
        $this->taskClient = $taskClient;
        $this->commandBus = $taskBus;
        $this->logger = new NullLogger();
    }

    /**
     * @param TaskRestfulModel $task
     * @return array
     */
    protected function run(TaskRestfulModel $task)
    {
        $this->logger->info('Lancement de la tâche n°' . $task->id);
        return $this->taskClient->taskRun($task);
    }

    /**
     * @param TaskRestfulModel $task
     * @param array $result
     * @return array
     */
    protected function success(TaskRestfulModel $task, $result)
    {
        $this->logger->info('La tâche n°' . $task->id . ' a été exécutée.');
        return $this->taskClient->taskSuccess($task, $result);
    }

    /**
     * @param TaskRestfulModel $task
     * @param array $result
     * @return array
     */
    protected function fail(TaskRestfulModel $task, $result)
    {
        $this->logger->info('La tâche n°' . $task->id . ' a échoué.');
        return $this->taskClient->taskFail($task, $result);
    }

    /**
     * @param TaskRestfulModel $task
     * @param LinkResourceModel $resource
     * @return array
     */
    protected function execCommand(TaskRestfulModel $task, LinkResourceModel $resource = null)
    {
        // On initialise les paramètres de retour
        $stderr = null;
        $resCode = 0;
        ob_start();
        try {
            // On récupère la commande
            $command = WorkerCommandFactory::actionTypeToCommand(
                $task->action_type,
                $resource,
                $task->action_params
            );

            // Puis on l'exécute
            $result = $this->commandBus->handle($command);
            if (!empty($result)) {
                echo serialize($result);
            }
        } catch (\Exception $e) {
            // On récupère le code et le message d'erreur
            $resCode = $e->getCode() != 0 ? $e->getCode() : 500;
            $this->logger->error($e->getMessage(), [
                'trace' => $e->getTrace(),
                'code' => $resCode,
            ]);
        }
        // On construit le tableau pour les logs
        $stdout = ob_get_clean();
        // lecture des erreurs du stream
        rewind($this->logStream);
        $stderr = stream_get_contents($this->logStream);
        // remise a zero du stream pour eviter les collision
        ftruncate($this->logStream, 0);
        // fabrique du log de retour
        $log = [
            'result_code' => (int)$resCode,
            'stdout' => $stdout,
            'stderr' => $stderr,
        ];
        return $log;
    }

    /**
     * @param TaskRestfulModel $task
     * @throws \UnexpectedValueException
     */
    protected function execTask(TaskRestfulModel $task)
    {
        // On initialise les logs
        $log = [
            'result_code' => 0,
            'stdout' => '',
            'stderr' => null,
        ];
        // On exécute la commande pour chaque ressource
        $resources = $task->getFlattenResources();
        if (empty($resources)) {
            $log = $this->execCommand($task);
        } else {
            foreach ($resources as $resource) {
                if (!$resource instanceof LinkResourceModel) {
                    continue;
                }
                $log = $this->execCommand($task, $resource);
                // On quitte la boucle si on a une des commandes échoue
                if ($log['result_code'] !== 0 && $log['stderr'] != null) {
                    break;
                }
            }
        }
        // on change le status de la tâche et si elle réussit, on change la date de la prochaine exécution
        if (empty($log) || $log['result_code'] === 0) {
            $this->success($task, $log);
        } else {
            $this->fail($task, $log);
        }
    }
}
