<?php

namespace Cms\Client\CmsUser;

use Cms\Client\AbstractClient;
use Cms\Model\CmsUser\Session\SessionModel;
use Move\Http\Client\Exception\HttpClientException;
use POM\DomainObjectInterface;

/**
 * Class CmsUserSessionClient
 * @package Cms\Client\CmsUser
 */
class CmsUserSessionClient extends AbstractClient
{

    /** @var string */
    protected $basePath = 'cms/user/session';

    /**
     * @param array $dataset
     * @return SessionModel
     */
    public function handleData($dataset = [])
    {
        return (new SessionModel())->populate($dataset);
    }

    /**
     * @param array $queryParams
     * @return \Cms\Client\CmsApiIndexIterator|\Cms\Model\CmsUser\Session\SessionModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndexWithParams($queryParams)
    {
        $result = $this->loadIndexFromClient($this->basePath, $queryParams);
        return $result;
    }

    /**
     * Log un utilisateur et renvoie la session créée
     * @param string $login
     * @param string $password
     * @return SessionModel
     * @throws \InvalidArgumentException
     * @throws \DomainException
     */
    public function loginUser($login, $password) : SessionModel
    {
        $sessionModel = new SessionModel();
        try {
            $result = $this->createObjectFromClient(
                $sessionModel,
                'cms/user/login',
                [
                    'email' => $login,
                    'password' => $password,
                ]
            );
        } catch (HttpClientException $e) {
            throw new \InvalidArgumentException('bad parameter passed : ' . $e->getMessage(), null, $e);
        } catch (\Exception $e) {
            throw new \DomainException('an error occured while login user : ' . $e->getMessage(), null, $e);
        }

        if (!$result instanceof SessionModel) {
            throw new \DomainException('unexpected session value');
        }
        return $result;
    }

    /**
     * Récupère la dernière session connue d'un utilisateur
     * @param int $userId
     * @return \Cms\Client\CmsApiIndexIterator|SessionModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getLastSessionByUserId($userId)
    {
        $results = $this->loadIndexFromClient('cms/user/' . $userId . '/session', [
            'last' => 1,
        ]);
        return $results;
    }

    /**
     * Supprime une session
     * @param DomainObjectInterface|SessionModel $object
     * @param string $primaryKey
     * @return bool
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function deleteItem(DomainObjectInterface $object, $primaryKey = 'id')
    {
        if (!$object instanceof SessionModel) {
            throw new \InvalidArgumentException('object has to be a SessionModel');
        }
        $result = parent::deleteItem($object, 'session_id');
        return $result;
    }

    /**
     * @param SessionModel $session
     * @return SessionModel|bool
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function createItem(SessionModel $session)
    {
        $session = $this->createObjectFromClient($session, 'cms/user/session');
        return $session;
    }
}
