<?php

namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\MemberNotFoundException;
use Cms\Client\Member\MemberClient;
use Cms\Client\Member\Model\MemberRestfulModel;
use League\Route\Http\Exception\UnauthorizedException;
use Move\Environment;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Trait MemberUtilsTrait
 * @package Cms\Bundle\Member
 */
trait MemberUtilsTrait
{
    /** @var  MemberRestfulModel */
    protected $memberModel;


    /**
     * @param string $email
     * @return string
     */
    protected function resolvEmail(string $email) : string
    {
        // envoi du mail
        if (!preg_match('#@newcom-regie\.com$#', $email) && (new Environment())->isDev()) {
            $email = 'devteam@ewcom-regie.com';
        }
        return $email;
    }

    /**
     * @param MemberClient $memberClient
     * @param $identifier
     * @param bool $nocache
     * @return \Cms\Client\Member\Model\MemberRestfulModel
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     */
    protected function getMemberModel(
        MemberClient $memberClient,
        $identifier,
        bool $nocache = false
    ) : MemberRestfulModel {
        // récuperation du cache
        if ($nocache === false
            && $this->memberModel instanceof MemberRestfulModel
            && ($this->memberModel->email === $identifier
                || (int)$this->memberModel->id === $identifier)
        ) {
            return $this->memberModel;
        }
        $memberModel = null;
        try {
            // récuperation des infos User
            if (is_numeric($identifier)) {
                $memberModel = $memberClient->fetchItem($identifier, [], $nocache);
            } elseif (filter_var($identifier, FILTER_VALIDATE_EMAIL)) {
                $memberModel = $memberClient->getItemByEmail($identifier, $nocache);
            } else {
                throw new \InvalidArgumentException('bad member identifier ' . $identifier);
            }
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot get member info from API',
                $e,
                MemberErrorCode::FETCH_MEMBER_API_FAILED
            );
        }
        if (!$memberModel instanceof MemberRestfulModel) {
            throw new MemberNotFoundException($identifier);
        }
        return $this->memberModel = $memberModel;
    }

    /**
     * @param ServerRequestInterface $request
     * @param $memberId
     * @return bool
     */
    protected function checkToken(ServerRequestInterface $request, &$memberId) : bool
    {
        try {
            // check du token décodé
            $idFromToken = MemberHelper::decodeToken($request);
            if ($idFromToken && is_numeric($idFromToken)) {
                $memberId = $idFromToken;
                return true;
            }
        } catch (UnauthorizedException $e) {
            return false;
        }

        return false;
    }
}
