<?php


namespace Cms\LinkResource\Provider\CMS;

use Cms\Client\Service\AddressBookService;
use Cms\Client\Service\ArticleService;
use Cms\Client\Service\CampsiteService;
use Cms\Client\Service\GeodataService;
use Cms\Client\Service\MemberService;
use Cms\Search\Article\ArticleSearchClient;
use Cms\Search\Campsite\CampsiteSearchClient;
use Interop\Container\ContainerInterface;
use Move\Cache\CacheAwareInterface;
use Move\Cache\CacheAwareTrait;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

/**
 * Class CmsProviderFactory
 * @package Cms\LinkResource\Provider\CMS
 */
class CmsProviderFactory implements CacheAwareInterface, LoggerAwareInterface
{
    use CacheAwareTrait;
    use LoggerAwareTrait;

    /**
     * @var ArticleService
     */
    private $articleService;

    /** @var ArticleSearchClient */
    private $articleSearchClient;

    /** @var  AddressBookService */
    private $bookService;

    /** @var MemberService */
    private $memberService;

    /** @var \Cms\Client\Service\GeodataService|null */
    private $geoService;

    /** @var \Cms\Client\Service\CampsiteService|null */
    private $campsiteService;

    /** @var \Cms\Search\Campsite\CampsiteSearchClient|null */
    private $campsiteSearchClient;

    /**
     * @param \Interop\Container\ContainerInterface $container
     * @return CmsProviderFactory
     * @throws \Psr\Container\NotFoundExceptionInterface
     * @throws \Psr\Container\ContainerExceptionInterface
     */
    public static function createWithContainer(ContainerInterface $container) : CmsProviderFactory
    {
        $articleService = null;
        if ($container->has(ArticleService::class)) {
            $articleService = $container->get(ArticleService::class);
        }
        $articleSearchClient = null;
        if ($container->has(ArticleSearchClient::class)) {
            $articleSearchClient = $container->get(ArticleSearchClient::class);
        }
        $bookService = null;
        if ($container->has(AddressBookService::class)) {
            $bookService = $container->get(AddressBookService::class);
        }
        $memberService = null;
        if ($container->has(MemberService::class)) {
            $memberService = $container->get(MemberService::class);
        }
        $geoService = null;
        if ($container->has(GeodataService::class)) {
            $geoService = $container->get(GeodataService::class);
        }
        $campisteService = null;
        if ($container->has(CampsiteService::class)) {
            $campisteService = $container->get(CampsiteService::class);
        }
        $campsiteSearchClient = null;
        if ($container->has(CampsiteSearchClient::class)) {
            $campsiteSearchClient = $container->get(CampsiteSearchClient::class);
        }

        $providerFactory = new self(
            $articleService,
            $articleSearchClient,
            $bookService,
            $memberService,
            $campisteService,
            $campsiteSearchClient,
            $geoService
        );

        // ajoute le cache pool
        if ($container->has(CacheItemPoolInterface::class)) {
            $pool = $container->get(CacheItemPoolInterface::class);
            $providerFactory->setItemCachePool($pool);
        }

        // ajoute le logger
        if ($container->has(LoggerInterface::class)) {
            $logger = $container->get(LoggerInterface::class);
            $providerFactory->setLogger($logger);
        }

        return $providerFactory;
    }

    /**
     * CmsProviderFactory constructor.
     * @param ArticleService $articleService
     * @param ArticleSearchClient $articleSearchClient
     * @param AddressBookService $bookService
     * @param MemberService $memberService
     * @param \Cms\Client\Service\CampsiteService|null $campsiteService
     * @param \Cms\Search\Campsite\CampsiteSearchClient|null $campsiteSearchClient
     * @param \Cms\Client\Service\GeodataService|null $geoService
     */
    public function __construct(
        ArticleService $articleService = null,
        ArticleSearchClient $articleSearchClient = null,
        AddressBookService $bookService = null,
        MemberService $memberService = null,
        CampsiteService $campsiteService = null,
        CampsiteSearchClient $campsiteSearchClient = null,
        GeodataService $geoService = null
    ) {
        $this->articleService = $articleService;
        $this->articleSearchClient = $articleSearchClient;
        $this->bookService = $bookService;
        $this->memberService = $memberService;
        $this->campsiteService = $campsiteService;
        $this->geoService = $geoService;
        $this->campsiteSearchClient = $campsiteSearchClient;
    }

    /**
     * @param GeodataService $geodataService
     * @return CmsGeodataProvider|null
     */
    public function createCmsGeodataProvider(
        GeodataService $geodataService = null
    ) {
        $geodataService = $geodataService ?: $this->geoService;
        if (!$geodataService) {
            return null;
        }
        $provider = new CmsGeodataProvider(
            $geodataService
        );
        return $provider;
    }

    /**
     * @param null|int|string $scopeId
     * @param \Cms\Client\Service\CampsiteService|null $campsiteService
     * @param \Cms\Search\Campsite\CampsiteSearchClient|null $campsiteSearchClient
     * @param \Cms\Client\Service\AddressBookService|null $bookService
     * @return \Cms\LinkResource\Provider\CMS\CmsCampsiteProvider|null
     */
    public function createCmsCampsiteProvider(
        $scopeId = null,
        CampsiteService $campsiteService = null,
        CampsiteSearchClient $campsiteSearchClient = null,
        AddressBookService $bookService = null
    ) {
        $campsiteService = $campsiteService ?: $this->campsiteService;
        if (!$campsiteService) {
            return null;
        }
        $provider = new CmsCampsiteProvider(
            $campsiteService,
            $campsiteSearchClient ?: $this->campsiteSearchClient,
            $this->createCmsAddressBookProvider($scopeId, $bookService),
            $this->getItemCachePool(),
            $this->logger
        );
        if (!empty($scopeId)) {
            $provider->setScopeId($scopeId);
        }
        return $provider;
    }

    /**
     * @param null|int $scopeId
     * @param ArticleService $articleService
     * @param ArticleSearchClient $searchClient
     * @return CmsArticleProvider|null
     */
    public function createCmsArticleProvider(
        $scopeId = null,
        ArticleService $articleService = null,
        ArticleSearchClient $searchClient = null
    ) {
        $articleService = $articleService ?: $this->articleService;
        if (!$articleService) {
            return null;
        }
        $provider = new CmsArticleProvider(
            $articleService,
            $searchClient ?: $this->articleSearchClient
        );
        if (!empty($scopeId)) {
            $provider->setScopeId($scopeId);
        }
        return $provider;
    }

    /**
     * @param null|int $scopeId
     * @param AddressBookService $bookService
     * @return CmsAddressBookProvider|null
     */
    public function createCmsAddressBookProvider(
        $scopeId = null,
        AddressBookService $bookService = null
    ) {
        $bookService = $bookService ?: $this->bookService;
        if (!$bookService) {
            return null;
        }
        $provider = new CmsAddressBookProvider($bookService);
        if (!empty($scopeId)) {
            $provider->setScopeId($scopeId);
        }
        return $provider;
    }

    /**
     * @param MemberService|null $memberService
     * @return CmsMemberProvider|null
     */
    public function createCmsMemberProvider(MemberService $memberService = null)
    {
        $memberService = $memberService ?: $this->memberService;
        if (!$memberService) {
            return null;
        }
        $provider = new CmsMemberProvider($memberService);
        return $provider;
    }

    /**
     * @param null|int $scopeId
     * @return CmsProvider
     */
    public function createCmsProvider($scopeId = null) : CmsProvider
    {
        $cmsProvider = new CmsProvider();
        if (($provider = $this->createCmsArticleProvider($scopeId))) {
            $cmsProvider->delegate($provider);
        }
        if (($provider = $this->createCmsAddressBookProvider($scopeId))) {
            $cmsProvider->delegate($provider);
        }
        if (($provider = $this->createCmsMemberProvider())) {
            $cmsProvider->delegate($provider);
        }
        if (($provider = $this->createCmsCampsiteProvider($scopeId))) {
            $cmsProvider->delegate($provider);
        }
        if (($provider = $this->createCmsGeodataProvider())) {
            $cmsProvider->delegate($provider);
        }
        if ($scopeId) {
            $cmsProvider->setScopeId($scopeId);
        }
        return $cmsProvider;
    }

    /**
     * @param ArticleService $articleService
     * @return $this
     */
    public function setArticleService(ArticleService $articleService)
    {
        $this->articleService = $articleService;
        return $this;
    }

    /**
     * @param ArticleSearchClient $articleSearchClient
     * @return $this
     */
    public function setArticleSearchClient(
        ArticleSearchClient $articleSearchClient
    ) {
        $this->articleSearchClient = $articleSearchClient;
        return $this;
    }

    /**
     * @param AddressBookService $bookService
     * @return $this
     */
    public function setBookService(AddressBookService $bookService)
    {
        $this->bookService = $bookService;
        return $this;
    }

    /**
     * @param MemberService $memberService
     * @return $this
     */
    public function setMemberService(MemberService $memberService)
    {
        $this->memberService = $memberService;
        return $this;
    }

    /**
     * @param \Cms\Client\Service\GeodataService $geoService
     * @return $this
     */
    public function setGeoService(GeodataService $geoService)
    {
        $this->geoService = $geoService;
        return $this;
    }

    /**
     * @param \Cms\Client\Service\CampsiteService $campsiteService
     * @return $this
     */
    public function setCampsiteService(CampsiteService $campsiteService)
    {
        $this->campsiteService = $campsiteService;
        return $this;
    }

    /**
     * @param \Cms\Search\Campsite\CampsiteSearchClient $campsiteSearchClient
     * @return $this
     */
    public function setCampsiteSearchClient(CampsiteSearchClient $campsiteSearchClient)
    {
        $this->campsiteSearchClient = $campsiteSearchClient;
        return $this;
    }
}
