<?php

namespace Cms\Client\Campsite;

use Cms\Client\AbstractClient;
use Cms\Client\Campsite\Model\CampsiteRestfulModel;
use Cms\Client\Campsite\Model\ThematicRestfulModel;
use Cms\Client\CmsApiIndexIterator;

/**
 * Class ThematicClient
 * @package Cms\Client\Campsite
 */
class ThematicClient extends AbstractClient
{

    /** @var string */
    protected $basePath = 'campsite/thematic';

    /**
     * @param array $dataset
     * @return ThematicRestfulModel
     */
    public function handleData($dataset = []) : Model\ThematicRestfulModel
    {
        return (new ThematicRestfulModel())->populate($dataset, true);
    }

    /**
     * @param array $queryParams
     * @return CmsApiIndexIterator|\Cms\Client\Campsite\Model\ThematicRestfulModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndexWithParams($queryParams)
    {
        $result = $this->loadIndexFromClient($this->basePath, $queryParams);
        return $result;
    }

    /**
     * @param int $scopeId
     * @param string $search
     * @param string|array|null $type
     * @param int $pageNum
     * @return \Cms\Client\Campsite\Model\ThematicRestfulModel[]|\Cms\Client\CmsApiIndexIterator
     */
    public function getIndexBySearch(
        int $scopeId,
        string $search,
        $type = null,
        int $pageNum = 0
    ) : CmsApiIndexIterator {
        return $this->getBuilder()
            ->setFilter('scope_id', $scopeId)
            ->setFilter('type', $type)
            ->setFilter('search', $search)
            ->getIndex($pageNum);
    }

    /**
     * @param int $scopeId
     * @param string|array $type
     * @return \Cms\Client\CmsApiIndexIterator|ThematicRestfulModel[]
     */
    public function getIndexByType(
        int $scopeId,
        $type
    ) : CmsApiIndexIterator {
        return $this->getBuilder()
            ->setFilter('scope_id', $scopeId)
            ->setFilter('type', $type)
            ->setSortCol('rank', 'asc')
            ->getIndex(-1);
    }

    /**
     * @param int $scopeId
     * @param array $divisionIds
     * @return \Cms\Client\CmsApiIndexIterator|null|\Cms\Client\Campsite\Model\ThematicRestfulModel[]
     */
    public function getIndexByDivision(int $scopeId, array $divisionIds)
    {
        return $this->getBuilder()
            ->setFilter('scope_id', $scopeId)
            ->setFilter('division_id', $divisionIds)
            ->getIndex(-1);
    }

    /**
     * @param \Cms\Client\Campsite\Model\CampsiteRestfulModel $campsite
     * @return \Cms\Client\CmsApiIndexIterator|null|\Cms\Client\Campsite\Model\ThematicRestfulModel[]
     */
    public function getIndexByCampsite(CampsiteRestfulModel $campsite)
    {
        // resolution des thematique associés
        $featureCodes = array_map(function ($feature) {
            return $feature['feature_code'];
        }, $campsite->features);
        $groupIds = array_map(function ($group) {
            return $group['group_id'];
        }, $campsite->groups);

        return $this->getBuilder()
            ->setFilter('scope_id', $campsite->scope_id)
            ->setFilter('division_id', [])
            ->setFilter('feature_code', $featureCodes)
            ->setFilter('group_id', $groupIds)
            ->getIndex(-1);
    }

    /**
     * @param int $scope
     * @param string $slug
     * @return \Cms\Client\Campsite\Model\ThematicRestfulModel|null|\POM\DomainObjectInterface
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     */
    public function fetchItemBySlug(int $scope, string $slug)
    {
        $object = $this->fetchItem($slug, [
            'scope' => $scope,
        ]);
        return $object;
    }
}
