<?php

namespace Move\Http\Strategy\Handler;

use GuzzleHttp\Psr7\Response;
use Move\Http\Strategy\Serializer\JsonSerializer;
use Move\Http\Strategy\Serializer\SerializerInterface;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class JsonHandlerServiceTest
 * @package Move\Http\Strategy\Handler
 */
class JsonHandlerServiceTest extends TestCase
{

    protected function createInstance()
    {
        // Should use mock...
        $serializer = new JsonSerializer();
        return new JsonHandlerService($serializer);
    }

    /**
     * @return array
     */
    public function provideIsSatisfiedBy()
    {
        return [
            ['text/html, application/html' ,false],
            ['application/json' ,true]
        ];
    }

    /**
     * @covers JsonHandlerService::__construct
     */
    public function testConstruct()
    {
        $instece = $this->createInstance();
        $this->assertInstanceOf(JsonHandlerService::class, $instece);
    }

    /**
     * @covers JsonHandlerService::__construct
     * @expectedException \InvalidArgumentException
     */
    public function testConstructThrowsInvalidArgumentsException()
    {
        /** @var SerializerInterface $serializer */
        $serializer = $this->getMockBuilder(SerializerInterface::class)
            ->getMock();

        return new JsonHandlerService($serializer);
    }

    /**
     * @dataProvider provideIsSatisfiedBy
     *
     * @depends testConstruct
     * @covers JsonHandlerService::isSatisfiedBy
     * @param $returnValue
     * @param $expected
     */
    public function testIsSatisfiedBy($returnValue, $expected)
    {
        /** @var ServerRequestInterface $requeatInterface */
        $requeatInterface = $this->getMockBuilder(ServerRequestInterface::class)
            ->getMock();

        $requeatInterface->method('getHeaderLine')
            ->with('Accept')
            ->will($this->returnValue($returnValue));

        $service = $this->createInstance();

        $this->assertEquals($expected, $service->isSatisfiedBy($requeatInterface));
    }

    /**
     * @covers JsonHandlerService::determineResponse
     */
    public function testDetermineResponse()
    {
        $expectedContentType = 'application/json';

        // Should use mock...
        $responseInterface = new Response();

        $service = $this->createInstance();

        $alteredService = $service->determineResponse(
            ['data'=>['head' => 'body']],
            $responseInterface
        );

        $this->assertInstanceOf(ResponseInterface::class, $alteredService);

        $contentType = $alteredService->getHeader('content-type');

        $this->assertArraySubset([$expectedContentType], $contentType);
    }
}
