<?php

namespace Move\Image;

use PHPUnit\Framework\TestCase;

/**
 * Class ImageUtilsTest
 * @package Move\Image
 */
class ImageUtilsTest extends TestCase
{
    /** @var  string image raw */
    protected $raw;

    /** @var string image path */
    protected $path = __DIR__ . '/file/picture.jpg';

    protected $pathGif = __DIR__ . '/file/picture.gif';

    protected $pathPng = __DIR__ . '/file/picture.png';

    protected $pathNoIptc = __DIR__ . '/file/picture-no-iptc.jpg';

    /**
     * @inheritdocs
     */
    public function setUp()
    {
        if (!$this->raw) {
            $this->raw = file_get_contents($this->path);
        }
    }

    /**
     * @covers ImageUtils::getIptcData
     */
    public function testGetIptcData()
    {
        getimagesize($this->path, $info);
        $iptc = ImageUtils::getIptcData($info);
        $this->assertInstanceOf(Iptc::class, $iptc);

        $this->assertEquals('Dummy title', $iptc->title);
    }

    /**
     * @covers ImageUtils::getIptcData
     * @expectedException \InvalidArgumentException
     */
    public function testGetIptcDataThrowsException()
    {
        getimagesize($this->pathNoIptc, $info);
        ImageUtils::getIptcData($info);
    }

    /**
     * @covers ImageUtils::isAnimated
     */
    public function testIsAnimated()
    {
        $raw = file_get_contents($this->pathGif);
        $this->assertTrue(
            ImageUtils::isAnimated($raw),
            'Gif found as not animated'
        );

        $this->assertFalse(
            ImageUtils::isAnimated($this->raw),
            'Jpeg should not be found as animated'
        );
    }

    /**
     * @covers ImageUtils::createImage
     * @return resource
     */
    public function testCreateImageJpeg()
    {
        // test jpeg image
        $imageJpeg = ImageUtils::createImage($this->raw);

        $this->assertIsCorrectImage($imageJpeg);

        return $imageJpeg;
    }

    /**
     * Create a resource and call the passed function with the
     * new created resource
     *
     * @param callable $func
     * @throws \Exception
     */
    protected function runTestThrowsException(callable $func)
    {
        // Creer un fichier temporaire
        $fname = tempnam('mlib', 'TST');
        $fakeRes = fopen($fname, 'r+');
        $result = false;

        try {
            $func($fakeRes);
        } catch (\Exception $exception) {
            $result = $exception;
        } finally {
            // Supprimer le fichier temporaire
            fclose($fakeRes);
            unlink($fname);
        }

        if ($result) {
            throw $result;
        }
    }

    /**
     * Assert that it is a GD Ressource and not empty image
     * @param $resource
     */
    protected function assertIsCorrectImage($resource)
    {
        $this->assertInternalType(
            \PHPUnit_Framework_Constraint_IsType::TYPE_RESOURCE,
            $resource
        );

        $this->assertEquals(
            'gd',
            get_resource_type($resource),
            'The resource is not a GD image'
        );

        $this->assertGreaterThan(0, imagesx($resource));
    }

    /**
     * @covers ImageUtils::createImage
     */
    public function testCreateImagePng()
    {
        // test png image
        $pngImegeRaw = file_get_contents($this->pathPng);
        $pngImage = ImageUtils::createImage($pngImegeRaw, 'imgage/png');

        $this->assertIsCorrectImage($pngImage);
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testCreateImageThrowsException()
    {
        ImageUtils::createImage('qwertyuiop');
    }

    /**
     * @covers  ImageUtils::scaleImageCut
     * @depends testCreateImageJpeg
     * @param resource $image
     */
    public function testScaleImageCut($image)
    {
        $size = [
            'width' => intdiv(imagesx($image), 3),
            'height' => intdiv(imagesy($image), 2)
        ];

        // scale cut image
        $scaled = ImageUtils::scaleImageCut($image, $size);

        $this->assertIsCorrectImage($scaled);

        $scaledSize = [
            'width' => imagesx($scaled),
            'height' => imagesy($scaled)
        ];

        $this->assertEquals($size, $scaledSize);
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testScaleImageCutThrowsException()
    {
        $this->runTestThrowsException(function ($resource) {
            ImageUtils::scaleImageCut($resource);
        });
    }

    /**
     * @covers  ImageUtils::scaleImage
     * @depends testCreateImageJpeg
     */
    public function testScaleImage()
    {
        // create the image
        $image = ImageUtils::createImage($this->raw);

        $newWidth = intdiv(imagesx($image), 2);
        $newHeight = intdiv(imagesy($image), 2);

        // scale image
        $scaled = ImageUtils::scaleImage($image, $newWidth, $newHeight);

        $this->assertIsCorrectImage($scaled);

        $this->assertEquals(imagesx($scaled), $newWidth);
        $this->assertEquals(imagesy($scaled), $newHeight);
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testScaleImageThrowsInvalidArgumentException()
    {
        $this->runTestThrowsException(function ($resource) {
            ImageUtils::scaleImage($resource, 10);
        });
    }

    /**
     * @depends testCreateImageJpeg
     * @covers  ImageUtils::scaleImageWithFocus
     */
    public function testScaleImageWithFocus()
    {
        // create the image
        $image = ImageUtils::createImage($this->raw);

        $newWidth = intdiv(imagesx($image), 4);
        $newHeight = intdiv(imagesy($image), 4);

        // scale image
        $scaled = ImageUtils::scaleImageWithFocus(
            $image, $newWidth, $newHeight,
            [
                'width' => $newWidth,
                'height' => $newHeight
            ]
        );

        $this->assertIsCorrectImage($scaled);

        $this->assertEquals(imagesx($scaled), $newWidth);
        $this->assertEquals(imagesy($scaled), $newHeight);
    }

    /**
     * @covers ImageUtils::imageToString
     */
    public function testImageJpegToString()
    {
        $resource = imagecreatefromjpeg($this->path);
        $string = ImageUtils::imageToString($resource, 'image/jpeg');

        $image = \imagecreatefromstring($string);

        $this->assertIsCorrectImage($image);
    }

    /**
     * @covers ImageUtils::imageToString
     */
    public function testImagePngToString()
    {
        $resource = imagecreatefrompng($this->pathPng);
        $string = ImageUtils::imageToString($resource, 'image/png');

        $image = \imagecreatefromstring($string);

        $this->assertIsCorrectImage($image);
    }

    /**
     * @covers ImageUtils::createPlaceholder
     */
    public function testCreatePlaceholder()
    {
        $width = 10;
        $heignt = 10;
        $placeHolder = ImageUtils::createPlaceholder($this->pathPng, $width, $heignt);
        $this->assertIsCorrectImage($placeHolder);
        $this->assertEquals($width, imagesx($placeHolder));
        $this->assertEquals($heignt, imagesy($placeHolder));
    }

    /**
     * @covers ImageUtils::createPlaceholder
     * @expectedException \InvalidArgumentException
     */
    public function testCreatePlaceholderThrowsInvalidArgumentException()
    {
        ImageUtils::createPlaceholder('fakePathToImage',10,10);
    }
}
