<?php

namespace Move\Http\Client\Exception;

use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * Class HttpRequestException
 * @package Move\Http\Client\Exception
 */
class HttpRequestException extends \Exception
{
    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var ResponseInterface
     */
    private $response;

    /**
     * HttpRequestException constructor.
     * @param RequestInterface $request
     * @param ResponseInterface $response
     * @param string $customMessage
     * @param \Exception|null $previous
     */
    public function __construct(
        RequestInterface $request,
        ResponseInterface $response = null,
        $customMessage = null,
        \Exception $previous = null
    ) {
        parent::__construct(
            $customMessage ?: ($response ? $response->getReasonPhrase() : 'request failed'),
            $response ? $response->getStatusCode() : 0,
            $previous
        );
        $this->request = $request;
        $this->response = $response;
    }

    /**
     * @param \Psr\Http\Message\RequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface|null $response
     * @param string|null $customMessage
     * @param \Exception|null $previous
     * @return \Move\Http\Client\Exception\HttpClientException|\Move\Http\Client\Exception\HttpRequestException|\Move\Http\Client\Exception\HttpServerException
     */
    public static function create(
        RequestInterface $request,
        ResponseInterface $response = null,
        $customMessage = null,
        \Exception $previous = null
    ) {
        if ($response instanceof ResponseInterface && $response->getStatusCode()) {
            if ($response->getStatusCode() >= 500) {
                return new HttpServerException($request, $response, $customMessage, $previous);
            } elseif ($response->getStatusCode() >= 400) {
                return new HttpClientException($request, $response, $customMessage, $previous);
            }
        }
        return new HttpRequestException($request, $response, $customMessage, $previous);
    }

    /**
     * @return RequestInterface
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * @return ResponseInterface
     */
    public function getResponse()
    {
        return $this->response;
    }

    /**
     * @return bool
     */
    public function hasResponse()
    {
        return $this->response !== null;
    }
}
