<?php

namespace Move\Filter;

use Move\Specification\AndSpecification;
use Move\Specification\ArraySpecification;
use Move\Specification\EmptySpecification;
use Move\Specification\SpecificationInterface;
use Move\Utils\Reflection;
use PHPUnit\Framework\TestCase;

/**
 * Class FilterTest
 * @package Move\Filter
 */
class FilterTest extends TestCase
{
    /**
     * @covers Filter::__construct
     */
    public function testSimpleConstruct()
    {
        $instance = new Filter();
        $this->assertInstanceOf(Filter::class, $instance);
    }

    /**
     * @covers Filter::__construct
     */
    public function testAdvancedConstruct()
    {
        $filters = range(0,5);
        $modifiers = ['strtoupper'];
        $sanitizers = [];
        $pipe = null;
        $instance = new Filter($filters, $modifiers, $sanitizers, $pipe);
        $this->assertInstanceOf(Filter::class, $instance);
    }

    /**
     * @depends testSimpleConstruct
     * @covers Filter::spec
     */
    public function testSpec()
    {
        $instance = new Filter();

        /** @var SpecificationInterface $spec */
        $spec = $this
            ->getMockBuilder(ArraySpecification::class)
            ->getMock();

        $this->assertEquals(
            $instance,
            $instance->spec('fieldName', $spec)
        );

        $savedSpecs = Reflection::getValue($instance,'specs');
        $this->assertInstanceOf(
            ArraySpecification::class,
            reset($savedSpecs)
        );

        return $instance;
    }

    /**
     * @param $instance
     * @depends testSpec
     */
    public function testSpecSecondTime(Filter $instance)
    {
        /** @var SpecificationInterface $spec */
        $spec = $this
            ->getMockBuilder(EmptySpecification::class)
            ->getMock();

        // call the func
        $instance->spec('fieldName', $spec);

        // get the spec attribute
        $savedSpecs = Reflection::getValue($instance,'specs');

        $this->assertCount(1, $savedSpecs);

        $this->assertInstanceOf(
            AndSpecification::class,
            reset($savedSpecs)
        );
    }

    /**
     * @covers Filter::filter
     * @depends testSpec
     */
    public function testFilter()
    {
        $instance = new Filter();

        // test with bool
        $this->assertEquals(
            $instance,
            $instance->filter('field', false)
        );

        // test with a SpecificationInterface
        $spec = new ArraySpecification();
        $this->assertEquals(
            $instance,
            $instance->filter('field', $spec)
        );

        // test with a callable
        $callable = function($data){ return $data; };
        $this->assertEquals(
            $instance,
            $instance->filter('field', $callable)
        );

    }

    /**
     * @depends testSimpleConstruct
     * @covers Filter::modify
     */
    public function testModify()
    {
        $instance = new Filter();
        $field = 'fieldName';
        $func = 'strtoupper';
        $arg = ['arg'];

        $this->assertEquals(
            $instance,
            $instance->modify($field, $func, $arg)
        );

        $savedModifier = Reflection::getValue($instance, 'modifiers');

        $this->assertNotEmpty($savedModifier);
        $this->assertNotNull($savedModifier[$field]);
        $this->assertEquals($func, $savedModifier[$field]['callback']);
        $this->assertEquals($arg, $savedModifier[$field]['args']);
    }

    /**
     * @depends testSimpleConstruct
     * @covers Filter::sanitize
     */
    public function testSanitize()
    {
        $instance = new Filter();
        $field = 'fieldName';
        $filter = '';
        $options = [];

        $this->assertEquals(
            $instance,
            $instance->sanitize($field, $filter, $options)
        );

        $savedSanitizer = Reflection::getValue($instance, 'sanitizers');
        $this->assertNotEmpty($savedSanitizer);
        $this->assertNotNull($savedSanitizer[$field]);
    }

    /**
     * @depends testSimpleConstruct
     * @covers Filter::pipe
     */
    public function testPipe()
    {
        $instance = new Filter();
        $filter = new Filter();

        $this->assertEquals(
            $instance,
            $instance->pipe($filter)
        );

        $piped = Reflection::getValue($instance, 'pipe_filter');
        $this->assertEquals($filter, $piped);
    }

    /**
     * @expectedException \InvalidArgumentException
     * @depends testSimpleConstruct
     */
    public function testProcessWithException()
    {
        $filter = new Filter();
        $filter->process('not array');
    }

    /**
     * @covers Filter::__invoke
     * @depends testSimpleConstruct
     */
    public function testInvoke()
    {
        /** @var Filter|\PHPUnit_Framework_MockObject_MockObject $mock */
        $mock = $this
            ->getMockBuilder(Filter::class)
            ->setMethods(['process'])
            ->getMock();

        $expected = 'called';

        $mock
            ->expects($this->any())
            ->method('process')
            ->willReturn($expected);

        // test invoke
        $this->assertEquals($expected, $mock([]));
    }

    /**
     * @expectedException \InvalidArgumentException
     * @depends testSimpleConstruct
     */
    public function testProcessNotArray()
    {
        $filter = new Filter();
        $filter->process('not an array data');
    }

    /**
     * @depends testSimpleConstruct
     * @covers Filter::getFailedSpecs
     */
    public function testGetFailedSpecs()
    {
        $filter = new Filter();
        $this->assertEquals([], $filter->getFailedSpecs());
    }

    /**
     * @depends testSimpleConstruct
     * @covers Filter::getFailedValues
     */
    public function testGetFailedValue()
    {
        $filter = new Filter();
        $this->assertEquals([], $filter->getFailedValues());
    }

    /**
     * @depends testGetFailedSpecs
     * @depends testGetFailedValue
     */
    public function testProcess()
    {
        // todo implementation
        // need use case list
    }
}
