<?php

namespace Cms\Client\Campsite\Model;

use Cms\LinkResource\AbstractLinkResourceModel;
use Cms\LinkResource\FlattenResourcesAwareInterface;
use Cms\LinkResource\FlattenResourcesTrait;
use Cms\LinkResource\LinkResourceType;
use Cms\Model\Campsite\CampsiteAgendaModel;
use Cms\Model\Campsite\CampsiteAnnotationModel;
use Cms\Model\Campsite\CampsiteFeatureModel;
use Cms\Model\Campsite\CampsiteFeaturePriceModel;
use Cms\Model\Campsite\CampsiteGroupModel;
use Cms\Model\Campsite\CampsiteLinkResourceModel;
use Cms\Model\Campsite\CampsiteModel;
use Cms\Model\Campsite\CampsitePackageModel;
use Cms\Model\Campsite\CampsiteProductModel;
use Cms\Model\Campsite\Feature\FeatureModel;
use Cms\Model\Campsite\Group\GroupModel;
use Cms\Model\Campsite\Product\ProductModel;

/**
 * Class CampsiteRestfulModel
 * @package Cms\Client\Campsite\Model
 * @property bool $is_boosted
 * @property \Cms\Client\Campsite\Model\CampsiteFeatureRestfulModel[] $features
 * @property \Cms\Client\Campsite\Model\CampsiteGroupRestfulModel[] $groups
 * @property \Cms\Client\Campsite\Model\CampsitePackageRestfulModel[] $packages
 * @property \Cms\Client\Campsite\Model\CampsiteProductRestfulModel[] $products
 * @property \Cms\Model\Campsite\CampsiteAgendaModel[] $agendas
 * @property \Cms\Model\Campsite\CampsiteAnnotationModel[] $annotations
 * @property \Cms\Model\Campsite\CampsitePubModel[] $pubs
 * @property \Cms\Model\Campsite\CampsiteLogType $log_type
 * @property string $log_note
 */
class CampsiteRestfulModel extends CampsiteModel implements
    FlattenResourcesAwareInterface
{
    use FlattenResourcesTrait;

    /** @var bool */
    protected $is_boosted;

    /** @var  \Cms\Client\Campsite\Model\CampsiteFeatureRestfulModel[] */
    private $featureIdx;

    /** @var  \Cms\Client\Campsite\Model\CampsiteFeatureRestfulModel[] */
    protected $features;

    /** @var  \Cms\Client\Campsite\Model\CampsiteGroupRestfulModel[] */
    protected $groups;

    /** @var  \Cms\Client\Campsite\Model\CampsitePackageRestfulModel[] */
    protected $packages;

    /** @var  \Cms\Client\Campsite\Model\CampsiteProductRestfulModel[] */
    protected $products;

    /** @var  \Cms\Model\Campsite\CampsiteAnnotationModel[] */
    protected $annotations;

    /** @var  \Cms\Model\Campsite\CampsiteAgendaModel[] */
    protected $agendas;

    /** @var  \Cms\Model\Campsite\CampsitePubModel[] */
    protected $pubs;

    /** @var string */
    protected $log_note;

    /** @var \Cms\Model\Campsite\CampsiteLogType */
    protected $log_type;

    /**
     * @param string $identifier
     * @return \Cms\LinkResource\LinkResourceModel|null
     */
    public function getFirstAddress($identifier)
    {
        /** @noinspection ExceptionsAnnotatingAndHandlingInspection */
        $resources = $this->getResourceByType(
            LinkResourceType::ADDRESS_ADDRESS,
            'identifier',
            $identifier
        );
        if (!$resources) {
            return null;
        }
        return $resources[0];
    }

    /**
     * Renvoi les données avec le code de la feature en clé du tableau
     * @return \Cms\Client\Campsite\Model\CampsiteFeatureRestfulModel[]
     */
    public function getFeatureIndex() : array
    {
        $idx = $this->featureIdx
            ?? array_combine(
                array_map(
                    function (CampsiteFeatureRestfulModel $feature) {
                        return $feature->feature_code;
                    },
                    $this->features
                ),
                $this->features
            );
        $this->featureIdx = $idx;
        return $idx;
    }

    /**
     * @param string $featureCode
     * @return \Cms\Client\Campsite\Model\CampsiteFeatureRestfulModel|null
     */
    public function getFeatureByCode(string $featureCode)
    {
        $idx = $this->getFeatureIndex();
        return $idx[$featureCode] ?? null;
    }

    /**
     * @param string $featureCode
     * @return bool
     */
    public function hasFeature(string $featureCode) : bool
    {
        $idx = $this->getFeatureIndex();
        return isset($idx[$featureCode]);
    }

    /**
     * @param int $guideId
     * @param string|null $note
     * @return $this
     */
    public function addAnnotation($guideId, string $note = null) : CampsiteRestfulModel
    {
        if (is_numeric($guideId)) {
            $this->annotations[] = (new CampsiteAnnotationModel())->populate([
                'guide_id' => $guideId,
                'note' => $note,
            ], true);
        }
        return $this;
    }

    /**
     * @param \Cms\Model\Campsite\CampsiteAgendaModel $agendaModel
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     */
    public function addAgenda(CampsiteAgendaModel $agendaModel) : CampsiteRestfulModel
    {
        $this->agendas[] = $agendaModel;
        return $this;
    }

    /**
     * @param \Cms\Model\Campsite\CampsitePackageModel $packageModel
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     */
    public function addPackage(CampsitePackageModel $packageModel) : CampsiteRestfulModel
    {
        if (!$packageModel instanceof CampsitePackageRestfulModel) {
            $packageModel = (new CampsitePackageRestfulModel())
                ->populate($packageModel->getArrayCopy(), true);
        }
        $this->packages[] = $packageModel;
        return $this;
    }

    /**
     * @param \Cms\Model\Campsite\CampsiteProductModel $productModel
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     */
    public function addProduct(CampsiteProductModel $productModel) : CampsiteRestfulModel
    {
        $this->products = array_filter(
            $this->products,
            function (CampsiteProductRestfulModel $model) use ($productModel) {
                return $model->product_id !== $productModel->product_id;
            }
        );
        if (!$productModel instanceof CampsiteProductRestfulModel) {
            $productModel = (new CampsiteProductRestfulModel())
                ->populate($productModel->getArrayCopy(), true);
        }
        $this->products[] = $productModel;
        return $this;
    }

    /**
     * @param int|\Cms\Model\Campsite\Product\ProductModel $productId
     * @param string|null $value
     * @param string|null $note
     * @param int|null $quantity
     * @param bool $isFree
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     * @throws \InvalidArgumentException
     */
    public function setProduct(
        $productId,
        string $value = null,
        string $note = null,
        int $quantity = null,
        bool $isFree = true
    ) : CampsiteRestfulModel {
        if ($productId instanceof ProductModel) {
            $productId = $productId->id;
        }
        if (!is_numeric($productId)) {
            throw new \InvalidArgumentException('productId must be numeric or ProductModel');
        }
        $productModel = (new CampsiteProductRestfulModel())->populate([
            'product_id' => $productId,
            'value' => $value,
            'note' => $note,
            'quantity' => $quantity,
            'is_free' => $isFree,
        ], true);
        return $this->addProduct($productModel);
    }

    /**
     * @param \Cms\Model\Campsite\CampsiteGroupModel $groupModel
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     */
    public function addGroup(CampsiteGroupModel $groupModel) : CampsiteRestfulModel
    {
        $this->groups = array_filter(
            $this->groups,
            function (CampsiteGroupRestfulModel $model) use ($groupModel) {
                return $model->group_id !== $groupModel->group_id;
            }
        );
        if (!$groupModel instanceof CampsiteGroupRestfulModel) {
            $groupModel = (new CampsiteGroupRestfulModel())
                ->populate($groupModel->getArrayCopy(), true);
        }
        $this->groups[] = $groupModel;
        return $this;
    }

    /**
     * @param int|GroupModel $groupId
     * @param \DateTime|null $attachAt
     * @param string|null $note
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     * @throws \InvalidArgumentException
     */
    public function setGroup(
        $groupId,
        \DateTime $attachAt = null,
        string $note = null
    ) : CampsiteRestfulModel {
        if ($groupId instanceof GroupModel) {
            $groupId = $groupId->id;
        }
        if (!is_numeric($groupId)) {
            throw new \InvalidArgumentException('groupId must be numeric or GroupModel');
        }
        $groupModel = (new CampsiteGroupRestfulModel())->populate([
            'group_id' => $groupId,
            'attach_at' => $attachAt,
            'note' => $note,
        ], true);
        return $this->addGroup($groupModel);
    }

    /**
     * @param \Cms\Model\Campsite\CampsiteFeatureModel $featureModel
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     */
    public function addFeature(CampsiteFeatureModel $featureModel) : CampsiteRestfulModel
    {
        $this->features = array_filter(
            $this->features,
            function (CampsiteFeatureRestfulModel $model) use ($featureModel) {
                return $model->feature_code !== $featureModel->feature_code;
            }
        );
        if (!$featureModel instanceof CampsiteFeatureRestfulModel) {
            $featureModel = (new CampsiteFeatureRestfulModel())
                ->populate($featureModel->getArrayCopy(), true);
        }
        $this->features[] = $featureModel;
        return $this;
    }

    /**
     * @param string|\Cms\Model\Campsite\Feature\FeatureModel $featureCode
     * @param string|null $value
     * @param float|null $distance
     * @param string|null $distanceType
     * @param string|null $note
     * @param int|null $quantity
     * @param bool|null $isFree
     * @param \Cms\Model\Campsite\CampsiteFeaturePriceModel|null $price
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     * @throws \InvalidArgumentException
     */
    public function setFeature(
        $featureCode,
        string $value = null,
        float $distance = null,
        string $distanceType = null,
        string $note = null,
        int $quantity = null,
        bool $isFree = null,
        CampsiteFeaturePriceModel $price = null
    ) : CampsiteRestfulModel {
        if ($featureCode instanceof FeatureModel) {
            $featureCode = $featureCode->code;
        }
        if (!\is_string($featureCode)) {
            throw new \InvalidArgumentException('featureCode must be string or FeatureModel');
        }
        $featureModel = (new CampsiteFeatureRestfulModel())->populate([
            'feature_code' => $featureCode,
            'value' => $value,
            'distance' => $distance,
            'distance_type' => $distanceType,
            'note' => $note,
            'quantity' => $quantity,
            'is_free' => $isFree,
        ], true);
        if (null !== $price) {
            $featureModel->addPrice($price);
        }
        return $this->addFeature($featureModel);
    }

    /**
     * @param AbstractLinkResourceModel|array $resourceData
     * @return AbstractLinkResourceModel|CampsiteLinkResourceModel
     */
    public function handleResource($resourceData) : CampsiteLinkResourceModel
    {
        if (!$resourceData instanceof CampsiteLinkResourceModel) {
            $resourceData = (new CampsiteLinkResourceModel())->populate($resourceData, true);
        }
        return $resourceData;
    }
}
