<?php

namespace Move\Jato\Mapper;

use GuzzleHttp\Client;
use Move\Command\Exception\BadParameterException;
use Move\Http\Client\GuzzleClientAdapter;
use Move\Jato\JatoClient;
use Move\Jato\Model\VehicleModel;
use PHPUnit\Framework\TestCase;

/**
 * Class VehicleMapperTest
 * @package Move\Jato\Mapper
 */
class VehicleMapperTest extends TestCase
{
    /**
     * @return VehicleMapper
     */
    public function testConstruct() : VehicleMapper
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );
        $mapper = new VehicleMapper($client);
        $clientTest = $mapper->getClient();

        $this->assertInstanceOf(JatoClient::class, $clientTest, 'Not a JatoClient');
        $this->assertEquals($client, $clientTest, 'Client error');

        return $mapper;
    }

    /**
     * @depends testConstruct
     * @param VehicleMapper $mapper
     */
    public function testGetClient(VehicleMapper $mapper)
    {
        $client = $mapper->getClient();
        $this->assertInstanceOf(JatoClient::class, $client, 'Error getClient');
    }

    /**
     * @return array
     */
    public function handleDataProvider() : array
    {
        $data = '[
  {
    "databaseName": "SSCF_CS2002",
    "name": "France",
    "flagUrl": "https://sslphotos.jato.com/Flags/F.png",
    "vehicles": [
      {
        "vehicleId": 777513320170101,
        "vehicleUid": 7775133,
        "makeNameGlobal": "ASTON MARTIN",
        "makeNameLocal": "Aston Martin",
        "makeKey": "Aston Martin",
        "makeNameToDisplay": "Aston Martin",
        "modelNameGlobal": "DB11",
        "modelNameLocal": "DB11",
        "modelKey": "DB11",
        "modelNameToDisplay": "DB11",
        "modelYear": "2017",
        "modelYearToDisplay": "2017",
        "numberOfDoors": "2",
        "numberOfDoorsToDisplay": "2",
        "shortBodyName": "coupe",
        "bodyCode": "CO",
        "bodyCodeToDisplay": "CO",
        "vehiclePhotoUrl": null,
        "drivenWheels": "R",
        "drivenWheelsToDisplay": "rear",
        "price": 180673,
        "priceToDisplay": "£180,674",
        "fuelType": "P",
        "fuelTypeToDisplay": "premium unleaded",
        "derivative": "V12",
        "derivativeToDisplay": "V12",
        "engine": "5.2",
        "transmission": "A",
        "trimLevel": "-",
        "trimLevelToDisplay": "-",
        "seats": "4",
        "liters": 5.2,
        "manufacturerCode": "",
        "maximumPowerKw": 447,
        "maximumPowerhpPs": 608,
        "vehicleType": "C"
      }
    ]
  }
  ]';
        $return = json_decode($data, true);
        return [$return];
    }

    /**
     * @dataProvider handleDataProvider
     * @param array $data
     */
    public function testHandleData(array $data)
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );
        $mapper = new VehicleMapper($client);

        $data = $data['vehicles'];
        $return = [];
        foreach ($data as $vehicle) {
            $vehicleModel = $mapper->handleData($vehicle);
            $return[] = $vehicleModel;
        }
        $this->assertNotEmpty($return, 'Array empty');

        foreach ($return as $vehicleTest) {
            $this->assertInstanceOf(VehicleModel::class, $vehicleTest, 'Not VehicleModel');
        }
    }

    /**
     * @depends testConstruct
     * @param VehicleMapper $mapper
     */
    public function testFetchAllForBody(VehicleMapper $mapper)
    {
        $dataBase = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';
        $modelKey = 'DB11';
        $modelYear = '2017';
        $doorNumber = '2';
        $bodyCode = 'CO';
        $drivenWheels = 'R';

        $result = $mapper->fetchAllForBody(
            $dataBase, $makeKey, $modelKey, $modelYear, $doorNumber, $bodyCode, $drivenWheels
        );

        $this->assertInternalType('array', $result, 'Not an array');
        foreach ($result as $vehicleTest) {
            $this->assertInstanceOf(VehicleModel::class, $vehicleTest, 'Not VehicleModel');
        }
    }

    /**
     * @depends testConstruct
     * @param VehicleMapper $mapper
     */
    public function testFetchAllForBodyInvalid(VehicleMapper $mapper)
    {
        $this->expectException(BadParameterException::class);

        $dataBase = null;
        $makeKey = null;
        $modelKey = null;
        $modelYear = null;
        $doorNumber = null;
        $bodyCode = null;
        $drivenWheels = null;
        $mapper->fetchAllForBody(
            $dataBase, $makeKey, $modelKey, $modelYear, $doorNumber, $bodyCode, $drivenWheels
        );
    }

}