<?php


namespace Cms\Bundle\Sitemap;

use Move\I18n\I18nLanguageCode;

/**
 * Class SitemapWriter
 * @package Cms\Bundle\Sitemap
 */
class SitemapWriter
{

    /** @var  array */
    private $fhlist;

    /** @var array */
    private $langMap;

    /** @var string */
    private $header;

    /** @var string */
    private $footer;

    /**
     * SitemapWriter constructor.
     * @param array $langMap
     * @param string|null $header
     * @param string|null $footer
     */
    public function __construct(array $langMap = [], string $header = null, string $footer = null)
    {
        $this->langMap = $langMap ?: [I18nLanguageCode::FR];
        $this->header = $header ?: SitemapStruct::SITEMAP_HEADER;
        $this->footer = $footer ?: SitemapStruct::SITEMAP_FOOTER;
    }

    /**
     * @return \Generator
     * @throws \DomainException
     */
    public function open()
    {
        foreach ($this->langMap as $locale) {
            $fh = $this->fhlist[$locale] ?? null;
            if (!\is_resource($fh[0])) {
                $tmpfname = tempnam('/tmp', 'sitemap-index');
                if (!$tmpfname) {
                    throw new \DomainException('cannot create temporary index file');
                }
                $fh = [fopen($tmpfname, 'wb+'), $tmpfname];
                fwrite($fh[0], $this->header);
                $this->fhlist[$locale] = $fh;
            }

            yield $locale => $this->fhlist[$locale];
        }
    }

    /**
     * @return \Generator
     * @throws \DomainException
     */
    public function close()
    {
        foreach ($this->open() as $locale => $fh) {
            fwrite($fh[0], $this->footer);
            fclose($fh[0]);
            yield $locale => $fh[1];
        }
    }

    /**
     * @param resource $fh
     * @param string $url
     * @param string $lastMod
     * @param array $images
     * @throws \InvalidArgumentException
     */
    public static function writeloc($fh, string $url, string $lastMod, array $images = [])
    {
        if (!\is_resource($fh)) {
            throw new \InvalidArgumentException('resource is invalid');
        }
        $imageSet = '';
        foreach ($images as $image) {
            $imageSet .= /** @lang XML */
                <<<EOF
    <image:image>
      <image:loc>$image</image:loc>
    </image:image>
EOF;
        }

        $sitemapUrl = /** @lang XML */
            <<<EOF
<url>
    <loc>$url</loc>
    <lastmod>$lastMod</lastmod>
    $imageSet
</url>
EOF;
        fwrite($fh, $sitemapUrl . "\n");
    }
}
