<?php

namespace Cms\Client\Scheduler;

use Cms\Client\AbstractClient;
use Cms\Client\CmsApiIndexIterator;
use Cms\Client\Scheduler\Model\TaskRestfulModel;
use Cms\Model\Scheduler\Task\TaskModel;
use Move\Http\Client\HttpRequestOptions;

/**
 * Class SchedulerTaskClient
 * @package Cms\Client\Scheduler
 */
class SchedulerTaskClient extends AbstractClient
{
    /** @var string */
    protected $basePath = 'scheduler/task';

    /**
     * @param array $dataset
     * @return TaskRestfulModel
     */
    public function handleData($dataset = [])
    {
        return (new TaskRestfulModel())->populate($dataset, true);
    }

    /**
     * @param array $queryParams
     * @return CmsApiIndexIterator|TaskRestfulModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndexWithParams($queryParams)
    {
        $result = $this->loadIndexFromClient($this->basePath, $queryParams);
        return $result;
    }

    /**
     * @return TaskRestfulModel[]|CmsApiIndexIterator
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndex()
    {
        $result = $this->getIndexWithParams([]);
        return $result;
    }

    /**
     * @param $id
     * @return null|TaskRestfulModel
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getItem($id)
    {
        $user = $this->fetchItem($id);
        return $user;
    }

    /**
     * @param null|int|array $scopeId
     * @param int|null $limitRunning
     * @return \Cms\Client\Scheduler\Model\TaskRestfulModel[]|CmsApiIndexIterator
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpServerException
     */
    public function getTaskForExecution($scopeId = null, int $limitRunning = null)
    {
        $result = $this->loadIndexFromClient('scheduler/task/waiting', [
            'scope_id' => $scopeId,
            'filter' => [
                'limit_running' => $limitRunning,
            ],
        ]);
        return $result;
    }

    /**
     * @param TaskModel $task
     * @return array
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function taskRun(TaskModel $task)
    {
        return $this->request('POST', [sprintf('scheduler/task/%d/run', $task->id)]);
    }

    /**
     * @param TaskModel $task
     * @param array $result
     * @return array
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function taskSuccess(TaskModel $task, $result)
    {
        $options = [
            HttpRequestOptions::BODY_CONTENT => $result,
        ];
        return $this->request('POST', [sprintf('scheduler/task/%d/success', $task->id)], $options);
    }

    /**
     * @param TaskModel $task
     * @param array $result
     * @return array
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function taskFail(TaskModel $task, $result)
    {
        $options = [
            HttpRequestOptions::BODY_CONTENT => $result,
        ];
        return $this->request('POST', [sprintf('scheduler/task/%d/fail', $task->id)], $options);
    }
}
