<?php

namespace Move\Template\Html\Meta;

/**
 * Class Url
 * @package Html\Metadata
 */
class Url implements MetaInterface
{
    /** @var string */
    protected $canonical;

    /** @var null|string */
    protected $mobileurl;

    /** @var null|string */
    protected $rss;

    /** @var null|string */
    protected $atom;

    /** @var null|string */
    protected $publisher;

    /** @var  array */
    protected $langlist;

    /** @var  callable|null */
    protected $transformer;

    /** @var null|string */
    protected $amp;

    /**
     * Link Canonical/Alternate/Rss/Atom
     * @param string $canonical
     * @param null|string $mobileurl
     * @param null|string $rss
     * @param null|string $atom
     * @param null|string $publisher
     * @param null|string $amp
     */
    public function __construct(
        $canonical,
        $mobileurl = null,
        $rss = null,
        $atom = null,
        $publisher = null,
        $amp = null
    ) {
        $this->canonical = $canonical;
        $this->mobileurl = $mobileurl;
        $this->rss = $rss;
        $this->atom = $atom;
        $this->publisher = $publisher;
        $this->amp = $amp;
    }

    /**
     * @param string $canonical
     * @return $this
     */
    public function setCanonicalUrl($canonical)
    {
        $this->canonical = $canonical;
        return $this;
    }

    /**
     * @param null|string $mobileurl
     * @return $this
     */
    public function setMobileUrl($mobileurl)
    {
        $this->mobileurl = $mobileurl;
        return $this;
    }

    /**
     * @param null|string $rss
     * @return $this
     */
    public function setRssUrl($rss)
    {
        $this->rss = $rss;
        return $this;
    }

    /**
     * @param null|string $atom
     * @return $this
     */
    public function setAtomUrl($atom)
    {
        $this->atom = $atom;
        return $this;
    }

    /**
     * @param null|string $publisher
     * @return $this
     */
    public function setPublisherUrl($publisher)
    {
        $this->publisher = $publisher;
        return $this;
    }

    /**
     * @param null|string $amp
     * @return $this
     */
    public function setAmpUrl($amp)
    {
        $this->amp = $amp;
        return $this;
    }

    /**
     * @param array $langList
     * @param callable $transformer
     * @return $this
     */
    public function setAlternateLang(array $langList = [], callable $transformer = null)
    {
        $this->transformer = $transformer;
        $this->langlist = $langList;
        return $this;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return $this->render();
    }

    /**
     * @{@inheritdoc}
     */
    public function ident() : string
    {
        return 'link_url';
    }

    /**
     * @{@inheritdoc}
     */
    public function render() : string
    {
        $source = '';
        if ($this->canonical) {
            $source = '<link rel="canonical" href="' . $this->canonical . '">';
            if ($this->langlist && $this->transformer !== null) {
                foreach ($this->langlist as $lang) {
                    $alternateUrl = \call_user_func($this->transformer, $this->canonical, $lang);
                    if (\is_string($alternateUrl) && !empty($alternateUrl)) {
                        $source .= '<link rel="alternate" href="' . $alternateUrl . '" hreflang="' . $lang . '">';
                    }
                }
            }
        }
        if (null !== $this->rss && !empty($this->publisher)) {
            $source .= '<link rel="alternate" type="application/rss+xml" title="Version RSS" href="' . $this->rss . '">';
        }
        if (null !== $this->atom && !empty($this->publisher)) {
            $source .= '<link rel="alternate" type="application/atom+xml" title="Version Atom" href="' . $this->atom . '">';
        }
        if (null !== $this->publisher && !empty($this->publisher)) {
            $source .= '<link rel="publisher" href="' . $this->publisher . '">';
        }
        if (null !== $this->mobileurl && !empty($this->mobileurl)) {
            $source .= '<link rel="alternate" type="text/html" href="' . $this->mobileurl . '" media="only screen and (max-width: 640px)">';
        }
        if (null !== $this->amp && !empty($this->amp)) {
            $source .= '<link rel="amphtml" href="' . $this->amp . '" >';
        }
        return $source;
    }

    /**
     * Renvoi la copie sous forme de tableau de la metadata
     * @return array
     */
    public function getArrayCopy() : array
    {
        return [
            'canonical' => $this->canonical,
            'mobile' => $this->mobileurl,
            'rss' => $this->rss,
            'atom' => $this->atom,
            'publisher' => $this->publisher,
            'amp' => $this->amp,
        ];
    }
}
