<?php


namespace Cms\Client\Command\Article;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Command\Article\Command\AbstractFetchLinkedTypeCommand;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Service\ArticleService;
use Cms\LinkResource\LinkResourceType;
use Cms\Model\Article\LinkResource\LinkResourceModel;
use Move\Command\Exception\BadParameterException;
use Move\Http\Client\Exception\HttpRequestException;
use Psr\Cache\CacheItemInterface;

/**
 * Class FetchLinkedHandlerTrait
 * @package Cms\Client\Command\Article
 */
trait FetchLinkedTypeCommandHandlerTrait
{
    use CacheHelperTrait;
    use CommandExceptionHandleTrait;

    /**
     * @return ArticleService
     */
    abstract public function getArticleService() : ArticleService;

    /**
     * @param AbstractFetchLinkedTypeCommand $command
     * @return LinkResourceModel[]
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \InvalidArgumentException
     */
    public function handleFetchLinkedType(AbstractFetchLinkedTypeCommand $command) : array
    {
        $articleCatId = $command->getArticleCategoryId();
        $articleProfil = $command->getArticleProfil();
        $linkType = (string)$command->getLinkType();
        $scopeId = $command->getScopeId();

        // check type
        if (!LinkResourceType::isValid($linkType)) {
            throw new BadParameterException('link type is invalid : ' . $linkType);
        }

        $cacheKey = "$scopeId^"
            . implode('-', (array)$articleCatId)
            . "^$articleProfil^linked_" . $linkType
            . '^' . $command->hasMostUsed();

        /** @var CacheItemInterface $cacheItem */
        if (true === $this->hitFromCache($cacheKey, 3600, $cacheItem)) {
            return $cacheItem->get();
        }

        if (!empty($articleCatId)) {
            try {
                // récuperation des catégories
                $categoryClient = $this->getArticleService()->getCategoryClient();
                $categories = $categoryClient->getIndexByParentId($articleCatId, -1);
                $articleCatId = (array)$articleCatId;
                foreach ($categories as $childCategory) {
                    $articleCatId[] = $childCategory->id;
                }
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('CmdHd_FetchLinkedMarques', $e);
            }
        }

        try {
            $resourceClient = $this->getArticleService()->getLinkResourceClient();
            if (!empty($articleCatId)) {
                if ($command->hasMostUsed()) {
                    // récuperation des marques
                    $results = $resourceClient->getIndexMostUsedByCategory(
                        $articleCatId,
                        $linkType
                    );
                } else { // récuperation des marques
                    $results = $resourceClient->getIndexDistinctRefByCategory(
                        $articleCatId,
                        $linkType,
                        $articleProfil
                    );
                }
            } elseif ($command->hasMostUsed()) {
                $results = $resourceClient->getIndexMostUsed($scopeId, $linkType);
            } else {
                $results = $resourceClient->getIndexDistinctRef(
                    $scopeId,
                    $linkType,
                    $articleProfil
                );
            }
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('CmdHd_FetchLinkedMarques', $e);
        } finally {
            if (!isset($results)) {
                $results = [];
            }
        }

        $results = iterator_to_array($results, false);
        $this->setInCache($cacheItem, $results);

        return $results;
    }
}
