<?php

namespace Cms\Search;

use Cms\Search\Geodata\CityIndexFactory;
use Elastica\Client;
use Move\Search\AbstractIndexFactory;
use Move\Search\Analyzer\ElasticAnalyzer;
use Move\Search\Analyzer\GeoFrenchAnalyzer;
use Move\Search\Analyzer\GeoFrenchSearchAnalyzer;
use Move\Search\Analyzer\PredefinedFilterAnalyzer;
use Move\Search\ElasticIndexBuilder;
use Move\Search\ElasticIndexBuilderInterface;

/**
 * Class GeodataIndexFactory
 * @package Cms\Search
 */
class GeodataIndexFactory extends AbstractIndexFactory
{

    /** Nom generique de l'index dans es */
    public const INDEX_NAME = 'geodata';

    /**
     * GeodataIndexFactory constructor.
     * @param \Elastica\Client $elasticaClient
     * @param array $mappingBuilders
     * @throws \InvalidArgumentException
     */
    public function __construct(Client $elasticaClient, array $mappingBuilders = [])
    {
        parent::__construct($elasticaClient, $mappingBuilders ?: [
            CityIndexFactory::createMappingBuilder(),
        ]);
    }

    /**
     * @return ElasticAnalyzer
     */
    public static function createIndexAnalyzer() : ElasticAnalyzer
    {
        $indexAnalyzer = (new ElasticAnalyzer('indexAnalyzerFr', 'standard'))
            ->setFilter('french_elision', PredefinedFilterAnalyzer::FILTER_FRENCH_ELISION)
            ->setFilter('lowercase')
            ->setFilter('asciifolding')
            ->setFilter('french_stop', PredefinedFilterAnalyzer::FILTER_FRENCH_STOP)
            ->setFilter('autocomplete', PredefinedFilterAnalyzer::FILTER_AUTOCOMPLETE);
        return $indexAnalyzer;
    }

    /**
     * @return ElasticAnalyzer
     */
    public static function createSearchAnalyzer() : ElasticAnalyzer
    {
        $searchAnalyzer = (new ElasticAnalyzer('searchAnalyzerFr', 'standard'))
            ->setFilter('french_elision', PredefinedFilterAnalyzer::FILTER_FRENCH_ELISION)
            ->setFilter('asciifolding')
            ->setFilter('lowercase')
            ->setFilter('french_stop', PredefinedFilterAnalyzer::FILTER_FRENCH_STOP);
        return $searchAnalyzer;
    }

    /**
     * @param bool $recreate
     * @param int $shards
     * @return \Move\Search\ElasticIndexBuilder
     */
    public static function createIndexBuilder(
        bool $recreate = false,
        int $shards = null
    ) : ElasticIndexBuilderInterface {
        // calcul du nombre de replicas
        $replicas = (getenv('ELASTIC_CLUSTER_SIZE') ?: 1) - 1;

        // creation du builder
        $indexName = self::INDEX_NAME . '_' . date('dmY');
        $indexBuilder = new ElasticIndexBuilder($indexName, $shards ?? 2, $replicas ?: 2);
        if ($recreate === true) {
            $indexBuilder->forceRecreate();
        }

        // creation des analyzer
        $indexAnalyzer = self::createIndexAnalyzer();
        $searchAnalyzer = self::createSearchAnalyzer();

        // ajoute les analyzer
        $indexBuilder
            ->setAnalizer($indexAnalyzer)
            ->setAnalizer($searchAnalyzer)
            ->setAnalizer(new GeoFrenchAnalyzer())
            ->setAnalizer(new GeoFrenchSearchAnalyzer());

        // ajoute les alias
        $indexBuilder->setAliases([self::INDEX_NAME]);

        // creation de l'index
        return $indexBuilder;
    }

    /**
     * @return string
     */
    public static function getIndexName() : string
    {
        return self::INDEX_NAME;
    }
}
