<?php

namespace Move\Http\Middleware;

use Move\Utils\Str;
use Psr\Http\Message\MessageInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Message\StreamInterface;

/**
 * Class BodyParser
 * @package Move\Http\Middleware
 */
class BodyParser implements MiddlewareInterface
{

    /**
     * @param ServerRequestInterface $request
     * @return ServerRequestInterface
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function __invoke(ServerRequestInterface $request)
    {
        return $this->handle($request);
    }


    /**
     * Handle the payload.
     * @param ServerRequestInterface $request
     * @return ServerRequestInterface
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function handle(ServerRequestInterface $request)
    {
        $excludMethods = ['POST', 'PUT'];
        if ($request->getParsedBody() || !\in_array($request->getMethod(), $excludMethods, true)) {
            return $request;
        }
        $request = $request->withParsedBody($this->parseMessage($request));

        return $request;
    }

    /**
     * @param MessageInterface $message
     * @return array
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function parseMessage(MessageInterface $message)
    {
        if (!($body = $message->getBody()) || !$body->getContents()) {
            return [];
        }

        //form data
        $contentType = trim($message->getHeaderLine('Content-Type'));
        if (!empty($contentType)) {
            if (stripos($contentType, 'multipart/form-data') === 0) {
                return $this->parseFormData($body, $contentType);
            }
            //json
            if (stripos($contentType, 'application/json') === 0
                || stripos($contentType, 'text/javascript') === 0
            ) {
                return $this->parseJson($body);
            }
            //urlencoded
            if (stripos($contentType, 'application/x-www-form-urlencoded') === 0) {
                return $this->parseUrlEncoded($body);
            }
            //csv
            if (stripos($contentType, 'text/csv') === 0) {
                return $this->parseCsv($body);
            }
        }

        throw new \InvalidArgumentException(
            "this content type ($contentType) cannot be parse on message with content : " . $body
        );
    }

    /**
     * @see https://gist.github.com/jas-/5c3fdc26fedd11cb9fb5#file-stream-php
     * @param StreamInterface $body
     * @param string $contentType
     * @return array
     */
    protected function parseFormData(StreamInterface $body, $contentType)
    {
        $data = [];
        // read incoming data
        $input = (string)$body;

        // grab multipart boundary from content type header
        preg_match('/boundary=(.*)$/', $contentType, $headers);
        $boundary = $headers[1];

        // split content by boundary and get rid of last -- element
        $blocks = preg_split("/-+$boundary/", $input);
        array_pop($blocks);

        // loop data blocks
        foreach ($blocks as $id => $block) {
            if (empty($block)) {
                continue;
            }

            // parse uploaded files
            if (strpos($block, 'application/octet-stream') !== false) {
                // match "name", then everything after "stream" (optional) except for prepending newlines
                $stream = [];
                preg_match('/name="([^"]*)".*stream[\n|\r]+([^\n\r].*)?[\r|\n]+$/s', $block, $stream);

                $data[$stream[1]] = $stream[2];
            } // parse all other fields
            else {
                // match "name" and optional value in between newline sequences
                $matches = [];
                preg_match('/name="([^"]*)"' // get field name
                    . '(?:; filename="([^"]+)")?[\n|\r]+' // get filename if needed
                    . '(?:Content-Type: ([^\n\r]+)?[\n|\r]+)?' // Keep content-type for checking if not text/plain
                    . '(?:Content-[^:]+: (?:[^\n\r]+)?[\n|\r]+)?' // forget all other Content-* Headers
                    . '([^\n\r].*)?' // Content
                    . '[\r|\n]+$/s', $block, $matches);

                // exclude non text/plain values
                $value = trim($matches[4] ?? '');
                if (!empty($matches[3]) && $matches[3] !== 'text/plain') {
                    continue;
                    //$value = ['filename' => $matches[2], 'mime' => $matches[3], 'contents' => $value];
                }

                // if match name[(attr)?]
                $tmp = [];
                if (preg_match('/^(.*)\[([^\]]*)\]$/', $matches[1], $tmp)) {
                    if (empty($data[$tmp[1]])) {
                        $data[$tmp[1]] = [];
                    } elseif (!\is_array($data[$tmp[1]])) {
                        $data[$tmp[1]] = [$data[$tmp[1]]];
                    }

                    if (empty($tmp[2])) {
                        $data[$tmp[1]][] = $value;
                    } else {
                        $data[$tmp[1]][$tmp[2]] = $value;
                    }
                } else {
                    $data[$matches[1]] = $value;
                }
            }
        }

        return $data;
    }

    /**
     * Parses json.
     * @param StreamInterface $body
     * @return array
     */
    protected function parseJson(StreamInterface $body)
    {
        try {
            $data = Str::parseJson((string)$body, true);
            return $data;
        } catch (\Exception $e) {
            trigger_error($e->getMessage(), E_USER_WARNING);
        }
        return null;
    }

    /**
     * Parses url-encoded strings.
     * @param StreamInterface $body
     * @return array
     */
    protected function parseUrlEncoded(StreamInterface $body)
    {
        parse_str((string)$body, $data);
        return $data ?: [];
    }

    /**
     * Parses csv.
     * @param StreamInterface $body
     * @return array
     * @throws \RuntimeException
     */
    protected function parseCsv(StreamInterface $body)
    {
        if ($body->isSeekable()) {
            $body->rewind();
        }
        $stream = $body->detach();
        $data = [];
        while (($row = fgetcsv($stream)) !== false) {
            $data[] = $row;
        }
        fclose($stream);
        return $data;
    }
}
