<?php


namespace Move\Image;

/**
 * Class ImageUtils
 * @package Move\Image
 */
class ImageUtils
{

    /** @var int */
    public static $DEFAULT_QUALITY_PNG = 9;

    /** @var int */
    public static $DEFAULT_QUALITY_JPEG = 82;

    /**
     * @param array $imageinfo
     * @return Iptc
     * @throws \InvalidArgumentException
     */
    public static function getIptcData(array $imageinfo) : Iptc
    {
        return Iptc::createFromInfo($imageinfo);
    }

    /**
     * @param string $raw
     * @return bool
     */
    public static function isAnimated($raw) : bool
    {
        $offset = 0;
        $frames = 0;
        while ($frames < 2) {
            $where1 = strpos($raw, "\x00\x21\xF9\x04", $offset);
            if ($where1 === false) {
                break;
            }
            $offset = $where1 + 1;
            $where2 = strpos($raw, "\x00\x2C", $offset);
            if ($where2 === false) {
                break;
            }
            if ($where1 + 8 === $where2) {
                $frames++;
            }
            $offset = $where2 + 1;
        }

        return $frames > 1;
    }

    /**
     * @param string $content
     * @param string $mimeType
     * @return resource
     * @throws \InvalidArgumentException
     */
    public static function createImage($content, $mimeType = null)
    {
        $img = \imagecreatefromstring($content);
        if ($img === false) {
            throw new \InvalidArgumentException('content is not a valid image content');
        }
        if ($mimeType === null) {
            [, , $imageType] = getimagesizefromstring($content);
            $mimeType = image_type_to_mime_type($imageType);
        }
        // si png
        if ($mimeType === 'image/png') {
            imagealphablending($img, false);
            imagesavealpha($img, true);
        }
        return $img;
    }

    /**
     * @param resource $img
     * @param array $size
     * @return bool|null|resource
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public static function scaleImageCut($img, array $size = [])
    {
        if (get_resource_type($img) !== 'gd') {
            throw new \InvalidArgumentException('img must be a gd resource');
        }

        // recuperation des infos de l'image
        $imgWidth = imagesx($img);
        $imgHeight = imagesy($img);
        $ratio = $imgHeight / $imgWidth;

        // si une largeur
        if (!empty($size['width'])) {
            $pageWidth = $size['width'];
            // check des palier de largeur
            $pageHeight = $pageWidth * $ratio;
        }
        if (!empty($size['height'])) {
            if (empty($pageHeight) || $pageHeight < $size['height']) {
                $pageHeight = $size['height'];
            }
            // check des palier de hauteur

            // concervation du ratio d'origine
            $pageWidth = $pageHeight / $ratio;
            // si la largeur demandé est differente de la calculé on coupe les bord
            if (!empty($size['width']) && $pageWidth > $size['width']) {
                $cutWidth = $pageWidth - $size['width'];
            }
        }

        // scaling
        if (!empty($pageWidth) && !empty($pageHeight)) {
            $copyImg = static::scaleImage($img, $pageWidth, $pageHeight);
            $img = $copyImg;
        }

        // découpage des bords
        if (!empty($cutWidth)) {
            $copyImg = imagecrop($img, [
                'x' => (int)$cutWidth / 2,
                'y' => 0,
                'width' => $size['width'],
                'height' => $size['height'],
            ]);
            imagedestroy($img);
            return $copyImg;
        }
        return $img;
    }

    /**
     * @param resource $img
     * @param int $pageWidth
     * @param int|null $pageHeight
     * @return resource
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public static function scaleImage($img, $pageWidth, $pageHeight = -1)
    {
        if (get_resource_type($img) !== 'gd') {
            throw new \InvalidArgumentException('img must be a gd resource');
        }

        // scaling
        if (!empty($pageWidth) && !empty($pageHeight)) {
            // recuperation des infos de l'image
            $imgWidth = imagesx($img);
            $imgHeight = imagesy($img);
            // reseize de l'image
            $imgCopy = imagecreatetruecolor($pageWidth, $pageHeight);
            imagesetinterpolation($imgCopy, IMG_TRIANGLE);
            imagecopyresampled(
                $imgCopy,
                $img,
                0,
                0,
                0,
                0,
                $pageWidth,
                $pageHeight,
                $imgWidth,
                $imgHeight
            );
            imagedestroy($img);
            if (!$imgCopy) {
                throw new \RuntimeException("Echec de resize de l'image");
            }
            return $imgCopy;
        }
        return $img;
    }

    /**
     * @param resource $img
     * @param int $focusX
     * @param int $focusY
     * @param array $size
     * @return resource
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public static function scaleImageWithFocus($img, $focusX, $focusY, array $size = [])
    {
        if (get_resource_type($img) !== 'gd') {
            throw new \InvalidArgumentException('img must be a gd resource');
        }
        if (empty($size['width']) && empty($size['height'])) {
            return $img;
        }

        // recuperation des infos de l'image
        $pageWidth = $imgWidth = imagesx($img);
        $pageHeight = $imgHeight = imagesy($img);

        // calcul du ratio original de l'image
        $ratio = $imgHeight / $imgWidth;

        // si une largeur
        if (!empty($size['width'])) {
            $pageWidth = (int)$size['width'];
            // TODO : check des palier de largeur (palier de 50px)

            // concervation du ratio
            $pageHeight = $pageWidth * $ratio;
        }
        // si une hauteur
        if (!empty($size['height'])) {
            $pageHeight = (int)$size['height'];
            // TODO : check des palier de hauteur (palier de 50px)

            // concervation du ratio d'origine
            if (empty($size['width'])) {
                $pageWidth = $pageHeight / $ratio;
            }
        }

        // calcul de la taille de la page par rapport a la taille de l'image
        $pageRatio = $pageHeight / $pageWidth;
        if ($pageWidth > $imgWidth) {
            $pageWidth = $imgWidth;
            $pageHeight = $imgWidth * $pageRatio;
        } elseif ($pageHeight > $imgHeight) {
            $pageWidth = $imgHeight / $pageRatio;
            $pageHeight = $imgHeight;
        }

        // calcul de la taille de resize de l'image
        $pageResizeMode = false;
        if ($pageRatio > $ratio || ($pageRatio == $ratio && $ratio < 1)) {
            $resizeWidth = $pageHeight / $ratio;
            $resizeHeight = $pageHeight;
            $pageResizeMode = 'height';
        } elseif ($pageRatio < $ratio || ($pageRatio == $ratio && $ratio > 1)) {
            $resizeWidth = $pageWidth;
            $resizeHeight = $pageWidth * $ratio;
            $pageResizeMode = 'width';
        } else {
            $resizeWidth = $pageWidth;
            $resizeHeight = $pageHeight;
        }

        $yLeftCorner = $xLeftCorner = 0;

        // calcul de la taille du cadre
        // si hauteur ou largeur demandé plus grande que le scale on réduit la demande en suivant le ratio
        if (!empty($size['width']) && !empty($size['height'])) {
            // calcul du ratio entre coord des points et dimension de l'image
            $xRatio = ($focusX ?: 0) / $imgWidth;
            $yRatio = ($focusY ?: 0) / $imgHeight;

            // callage du point en haut a gauche par rapport a la page
            if ($pageRatio === 1) {
                if ($ratio < 1) {
                    $xLeftCorner = ($resizeWidth - $pageWidth) * $xRatio;
                    $yLeftCorner = 0;
                } elseif ($ratio > 1) {
                    $xLeftCorner = 0;
                    $yLeftCorner = ($resizeHeight - $pageHeight) * $yRatio;
                } else {
                    $xLeftCorner = 0;
                    $yLeftCorner = 0;
                }
            } else {
                $xLeftCorner = ($resizeWidth - $pageWidth) * $xRatio;
                $yLeftCorner = ($resizeHeight - $pageHeight) * $yRatio;

                // calcul du delta ideal
                $deltaHeight = abs($pageHeight * $yRatio - $pageHeight * .5);
                $deltaWidth = abs($pageWidth * $xRatio - $pageWidth * .5);

                // calcul du placemen de l'image si resize de largeur
                if ($pageResizeMode === 'width') {
                    if ($yRatio < .5) {
                        $yLeftCorner -= min(
                            abs($yLeftCorner), // delta en haut de l'image
                            $deltaHeight // calcul du delta ideal
                        );
                    } else {
                        $yLeftCorner += min(
                            abs($resizeHeight - $pageHeight - $yLeftCorner),
                            // calcul du delta en bas d'image
                            $deltaHeight // calcul du delta ideal
                        );
                    }
                }

                // calcul du placement de l'image si resize hauteur
                if ($pageResizeMode === 'height') {
                    if ($xRatio < .5) {
                        $xLeftCorner -= min(
                            abs($xLeftCorner), // delta a gauche de l'image
                            $deltaWidth // calcul du delta ideal
                        );
                    } else {
                        $xLeftCorner += min(
                            abs($resizeWidth - $pageWidth - $xLeftCorner),
                            // calcul du delta a droite d'image
                            $deltaWidth // calcul du delta ideal
                        );
                    }
                }
            }
        }
        /*
        var_dump($pageRatio, $ratio,
            'x:' . (int)$xLeftCorner,
            'y:' . (int)$yLeftCorner,
            'pw:' . (int)$pageWidth,
            'ph:' . (int)$pageHeight,
            'rw' . (int)$resizeWidth,
            'rh' . (int)$resizeHeight,
            'w' . (int)$imgWidth,
            'h' . (int)$imgHeight);
        exit();
        */

        // creation de l'image "cadre"
        $imageResult = imagecreatetruecolor((int)$pageWidth, (int)$pageHeight);

        // resize de l'image
        $imgCopy = static::scaleImage($img, (int)$resizeWidth, (int)$resizeHeight);

        // placement de l'image d'origine dans le cadre
        $result = imagecopyresampled(
            $imageResult,
            $imgCopy,
            0,
            0,
            (int)$xLeftCorner,
            (int)$yLeftCorner,
            (int)$resizeWidth,
            (int)$resizeHeight,
            (int)$resizeWidth,
            (int)$resizeHeight
        );
        imagedestroy($imgCopy);
        if (!$result) {
            throw new \RuntimeException("Echec de crop de l'image");
        }

        return $imageResult;
    }

    /**
     * @param resource $img
     * @param string $mimeType
     * @param null|int $quality
     * @return string
     */
    public static function imageToString($img, $mimeType, $quality = null) : string
    {
        // conversion de l'image en resource stream
        ob_start();
        if ($mimeType === 'image/png') {
            imagepng(
                $img,
                null,
                $quality && $quality <= 10 ? $quality : self::$DEFAULT_QUALITY_PNG
            );
        } elseif ($mimeType === 'image/jpeg') {
            imageinterlace($img, false);
            imagejpeg(
                $img,
                null,
                $quality && $quality <= 100 ? $quality : self::$DEFAULT_QUALITY_JPEG
            );
        }
        $content = ob_get_clean();
        imagedestroy($img);

        return $content;
    }

    /**
     * @param string $pngFilePath
     * @param null|int $width
     * @param null|int $height
     * @return resource
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public static function createPlaceholder(
        $pngFilePath,
        $width = null,
        $height = null
    ) {
        // check fichier placeholder
        if (!is_file($pngFilePath) || !is_readable($pngFilePath)) {
            throw new \InvalidArgumentException('PH filename is invalid');
        }

        // recuperation de l'image placeholder
        $img = imagecreatefrompng($pngFilePath);
        if (!$img) {
            throw new \UnexpectedValueException("cannot create image resource from file : $pngFilePath");
        }
        imagealphablending($img, true);

        // agrandissement du cadre ou est le ph
        if (!empty($width) && is_numeric($width)
            && !empty($height) && is_numeric($height)
        ) {
            $imgCopy = imagecreatetruecolor($width, $height);
            imagealphablending($imgCopy, false);
            $alpha = imagecolorallocatealpha($imgCopy, 255, 255, 255, 127);
            imagefilledrectangle($imgCopy, 0, 0, $width, $height, $alpha);
            imagealphablending($imgCopy, true);

            // placement du ph au millieu de l'image
            $xoffset = ($width - imagesx($img)) / 2;
            $yoffset = ($height - imagesy($img)) / 2;
            imagecopy($imgCopy, $img, $xoffset, $yoffset, 0, 0, imagesx($img), imagesy($img));
            imagedestroy($img);
            $img = $imgCopy;
        }

        imagealphablending($img, false);
        imagesavealpha($img, true);

        return $img;
    }
}
