<?php


namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\InvalidStateException;
use Cms\Bundle\Member\Exception\MemberNotFoundException;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberModel;
use Dflydev\FigCookies\FigResponseCookies;
use Dflydev\FigCookies\SetCookie;
use Move\Http\Client\Exception\HttpClientException;
use Psr\Http\Message\ResponseInterface;

/**
 * Trait MemberLoginTrait
 * @package Cms\Bundle\Member
 */
trait MemberLoginTrait
{

    /**
     * @return \Cms\Bundle\Member\MemberLoginService
     */
    abstract public function getLoginService() : MemberLoginService;

    /**
     * @param string $email
     * @param string $password
     * @param \Psr\Http\Message\ResponseInterface $response
     * @param bool $remembered
     * @return ResponseInterface
     * @throws \Cms\Bundle\Member\Exception\InvalidStateException
     * @throws \Cms\Bundle\Member\Exception\MemberNotFoundException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     */
    protected function withLoginPasswd(
        string $email,
        string $password,
        ResponseInterface $response,
        bool $remembered = false
    ) : ResponseInterface {
        try {
            /** @var MemberRestfulModel $result */
            $result = $this->getLoginService()->getMemberClient()
                ->loginMember($email, $password, $this->getLoginService()->getScopeId());
        } catch (HttpClientException $e) {
            if ($e->getMessage() === 'state') {
                throw new InvalidStateException($email);
            }
            $result = null;
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot get member datas',
                $e,
                MemberErrorCode::FETCH_MEMBER_API_FAILED
            );
        }

        if (!$result instanceof MemberModel) {
            throw new MemberNotFoundException($email);
        }

        return $this->withLogin($result, $response, $remembered);
    }

    /**
     * @param MemberModel $member
     * @param \Psr\Http\Message\ResponseInterface $response
     * @param bool $remembered
     * @return ResponseInterface
     */
    protected function withLogin(
        MemberModel $member,
        ResponseInterface $response,
        bool $remembered = false
    ) : ResponseInterface {
        // Création du token pour le cookie
        $datasForCookie = MemberHelper::createDataForCookie($member);

        // Envoie du cookie avec token/id/date
        if ($remembered) {
            $cookie = SetCookie::createRememberedForever(MemberHelper::COOKIE_MBRLOGIN);
        } else {
            $cookie = SetCookie::create(MemberHelper::COOKIE_MBRLOGIN);
        }
        $cookie = $cookie
            ->withHttpOnly(true)
            ->withPath('/')
            ->withValue(json_encode($datasForCookie));
        $response = FigResponseCookies::set($response, $cookie);

        return $response;
    }

    /**
     * @param \Psr\Http\Message\ResponseInterface $response
     * @return \Psr\Http\Message\ResponseInterface
     */
    protected function withLogout(
        ResponseInterface $response
    ) : ResponseInterface {
        $response = MemberHelper::removeCookie($response, MemberHelper::COOKIE_MBRLOGIN);
        return $response;
    }
}
