<?php


namespace Cms\Client\Geodata\Helper;

use Cms\Client\Command\Geodata\Command\FetchAdminCommand;
use Cms\Client\Command\Geodata\Command\FetchAllAdminByFilterCommand;
use Cms\Client\Command\Geodata\Command\FetchCityCommand;
use Cms\Client\Command\Geodata\Command\FetchCountryCommand;
use Cms\Client\Geodata\Model\AdminRestfulModel;
use Cms\Client\Service\GeodataService;
use Cms\Model\Geodata\CityModel;
use Cms\Model\Geodata\CountryModel;
use Cms\Model\Geodata\GeodataAwareModelInterface;
use Move\Command\Exception\NotFoundException;
use Move\ObjectMapper\ObjectTransformer;

/**
 * Class GeodataTransformerTrait
 * @package Cms\Client\Geodata
 */
trait GeodataTransformerTrait
{

    /**
     * @return \Cms\Client\Service\GeodataService
     * @throws \DomainException
     */
    abstract public function getGeodataService() : GeodataService;

    /**
     * @param GeodataAwareModelInterface $object
     * @return array
     * @throws \DomainException
     */
    public function transformAdminModel(GeodataAwareModelInterface $object) : array
    {
        $adminModel = $this->getAdminModel($object);
        return $adminModel ? (new ObjectTransformer())->transform($adminModel) : [];
    }

    /**
     * @param GeodataAwareModelInterface $object
     * @return array
     * @throws \DomainException
     */
    public function transformCountryModel(GeodataAwareModelInterface $object) : array
    {
        $countryModel = $this->getCountryModel($object);
        return $countryModel ? (new ObjectTransformer())->transform($countryModel) : [];
    }

    /**
     * @param GeodataAwareModelInterface $object
     * @return CountryModel|null
     * @throws \DomainException
     */
    public function getCountryModel(GeodataAwareModelInterface $object)
    {
        $geodataService = $this->getGeodataService();
        try {
            $command = new FetchCountryCommand($object->getCountryCode());
            $countryModel = $geodataService->getCommandBus()->handle($command);
            return $countryModel;
        } catch (NotFoundException $e) {
        }
        return null;
    }

    /**
     * @param GeodataAwareModelInterface $object
     * @return null|AdminRestfulModel
     * @throws \DomainException
     */
    public function getAdminModel(GeodataAwareModelInterface $object)
    {
        $geodataService = $this->getGeodataService();
        if ($object->getCityGeodataId()) {
            try {
                $command = new FetchCityCommand($object->getCityGeodataId());
                $geodataCity = $geodataService->getCommandBus()->handle($command);
            } catch (NotFoundException $e) {
            }

            if (!empty($geodataCity) && $geodataCity instanceof CityModel) {
                try {
                    $command = new FetchAdminCommand($geodataCity->admin2_code);
                    $resultAdmin = $geodataService->getCommandBus()->handle($command);
                } catch (NotFoundException $e) {
                }

                if (!empty($resultAdmin) && $resultAdmin instanceof AdminRestfulModel) {
                    return $resultAdmin;
                }
            }
        } elseif ($object->getAdmin2Code()) {
            try {
                $command = new FetchAllAdminByFilterCommand(
                    $object->getCountryCode(),
                    $object->getAdmin2Code(),
                    2
                );
                $admins = $geodataService->getCommandBus()->handle($command);
            } catch (NotFoundException $e) {
            }

            if (!empty($admins) && $admins->getTotal() >= 1) {
                $admins->rewind();
                return $admins->current();
            }
        }
        return null;
    }
}
