<?php

namespace Cms\Search\Provider\Elastica;

use Cms\Search\ResultSetInterface;
use Cms\Search\SearchClientAdapterInterface;
use Elastica\Client;
use Elastica\Query;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Class ElasticSearchAdapter
 * @package Cms\Search\Product
 */
class ElasticSearchAdapter implements SearchClientAdapterInterface, LoggerAwareInterface
{
    use LoggerAwareTrait;

    /**
     * @var Client
     */
    private $elasticaClient;

    /**
     * ElasticSearchAdapter constructor.
     * @param Client $elasticaClient
     */
    public function __construct(Client $elasticaClient)
    {
        $this->elasticaClient = $elasticaClient;
        $this->logger = new NullLogger();
    }

    /**
     * @param string $indexName
     * @param string $queryString
     * @param array $opts [type: string, sort: array]
     * @param int $limit
     * @param int $page
     * @return ElasticSearchResultSet|\Elastica\ResultSet|ResultSetInterface
     */
    public function searchString(
        string $indexName,
        string $queryString,
        array $opts = [],
        $limit = 10,
        $page = 1
    ) : ResultSetInterface {
        $cli = $this->elasticaClient->getIndex($indexName);
        if (!empty($opts['type']) && is_scalar($opts['type'])) {
            $cli = $cli->getType($opts['type']);
        }

        // clean des opts
        $requestOpts = $opts;
        unset(
            $requestOpts['options'],
            $requestOpts['sort'],
            $requestOpts['type'],
            $requestOpts['offset'],
            $requestOpts['size']
        );

        // creation de la requete
        $query = new Query([
            'query' => [
                'query_string' => array_merge($requestOpts, [
                    'query' => $queryString,
                ]),
            ],
        ]);

        // ajout des informations de trie
        if (!empty($opts['sort']) && \is_array($opts['sort'])) {
            $query->setSort($opts['sort']);
        }

        // applique une limite
        $limit = $limit ?: 10;
        $from = $opts['offset'] ?? 0;
        $query->setSize($opts['size'] ?? $limit);
        if ($from === 0 && $page >= 0) {
            $from = max(0, ($page - 1) * $limit);
        }
        $query->setFrom($from);

        $results = $cli->search(
            $query,
            $opts['options'] ?? null
        );
        $results = new ElasticSearchResultSet($results);
        return $results;
    }

    /**
     * @param string $indexName
     * @param mixed $query
     * @param array $opts
     * @param int $limit
     * @param int $page
     * @return ResultSetInterface
     */
    public function searchArray(
        string $indexName,
        array $query,
        array $opts = [],
        $limit = 10,
        $page = 1
    ) : ResultSetInterface {
        $cli = $this->elasticaClient->getIndex($indexName);
        if (!empty($opts['type']) && is_scalar($opts['type'])) {
            $cli = $cli->getType($opts['type']);
        }

        // clean des opts
        $requestOpts = $opts;
        unset(
            $requestOpts['options'],
            $requestOpts['sort'],
            $requestOpts['type'],
            $requestOpts['offset'],
            $requestOpts['size']
        );

        // creation de la requete
        $query = new Query(array_merge($requestOpts, $query));

        // ajout des informations de trie
        if (!empty($opts['sort']) && \is_array($opts['sort'])) {
            $query->setSort($opts['sort']);
        }

        // applique une limite
        $limit = $limit ?: 10;
        $from = $opts['offset'] ?? 0;
        $query->setSize($opts['size'] ?? $limit);
        if ($from === 0 && $page >= 0) {
            $from = max(0, ($page - 1) * $limit);
        }
        $query->setFrom($from);

        $results = $cli->search(
            $query,
            $opts['options'] ?? null
        );
        $results = new ElasticSearchResultSet($results);
        return $results;
    }
}
