<?php

namespace Cms\Bundle\Member;

use Cms\Model\Member\MemberModel;
use Dflydev\FigCookies\FigResponseCookies;
use Dflydev\FigCookies\SetCookie;
use League\Route\Http\Exception\UnauthorizedException;
use Move\Http\RequestUtils;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\JsonResponse;

/**
 * Class MemberHelper
 * @package Cms\Bundle\Member
 */
class MemberHelper
{
    public const CONCAT_CHAR = '::::';
    public const COOKIE_MBRLOGIN = '_MBRLGN';
    public const COOKIE_STATEAUTH = '_STATEAUTH';

    /**
     * @param MemberModel $member
     * @param string $ts
     * @return string
     */
    public static function encodeToken(MemberModel $member, $ts) : string
    {
        // encodage du token
        $token = base64_encode(
            password_hash(
                $member->id . self::CONCAT_CHAR . $ts,
                PASSWORD_BCRYPT
            )
        );
        return $token;
    }

    /**
     * @param ServerRequestInterface $request
     * @return int
     * @throws UnauthorizedException
     */
    public static function decodeToken(ServerRequestInterface $request) : int
    {
        // récupération du cookie
        $cookieValue = RequestUtils::GetCookieValue($request, '_MBRLGN');
        // check si le cookie existe
        if ($cookieValue) {
            $cookieDatas = json_decode($cookieValue, true);
            // vérification du token
            $decodedToken = base64_decode($cookieDatas['token']);
            if (password_verify(
                $cookieDatas['id'] . self::CONCAT_CHAR . $cookieDatas['ts'],
                $decodedToken
            )) {
                return (int)$cookieDatas['id'];
            }
        }
        // Sinon on envoie une 401
        throw new UnauthorizedException();
    }

    /**
     * @param ResponseInterface $response
     * @param string $name
     * @return ResponseInterface|JsonResponse
     * @internal param ServerRequestInterface $request
     */
    public static function removeCookie($response, $name)
    {
        // création d'un cookie expiré
        $response = FigResponseCookies::modify($response, $name, function (SetCookie $cookie) {
            $cookie = $cookie
                ->withHttpOnly(true)
                ->withPath('/')
                ->expire();
            return $cookie;
        });
        return $response;
    }

    /**
     * @param MemberModel $member
     * @return array
     */
    public static function createDataForCookie(MemberModel $member)
    {
        // récupèration du timestamp à la création
        $date = new \DateTime();
        $ts = $date->getTimestamp();
        // encodage du token
        $token = MemberHelper::encodeToken($member, $ts);
        // tableau pour cookie
        $datas = [
            'token' => $token,
            'id' => $member->id,
            'ts' => $ts,
        ];
        return $datas;
    }
}
