<?php

namespace Cms\Client\Asset;

use Cms\Client\AbstractClient;
use Cms\Client\Asset\Model\AssetResfulModel;
use Cms\Client\CmsApiIndexIterator;
use Cms\Model\Asset\AssetModel;
use Move\Http\Client\Exception\HttpClientException;
use Move\Http\Client\HttpRequestOptions;
use Move\Http\RequestUtils;
use Psr\Http\Message\StreamInterface;

/**
 * Class AssetClient
 * @package Cms\Client\Asset
 */
class AssetClient extends AbstractClient
{

    /** @var string */
    protected $basePath = 'asset';

    /**
     * @param array $dataset
     * @return AssetResfulModel
     */
    public function handleData($dataset = []) : AssetResfulModel
    {
        return (new AssetResfulModel())->populate($dataset, true);
    }

    /**
     * @param array $queryParams
     * @return CmsApiIndexIterator|AssetResfulModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndexWithParams($queryParams)
    {
        $result = $this->loadIndexFromClient($this->basePath, $queryParams);
        return $result;
    }

    /**
     * @param null|int $scope
     * @param int $page
     * @return AssetResfulModel[]|CmsApiIndexIterator
     */
    public function getIndex($scope = null, $page = 0)
    {
        $result = $this->getBuilder()->setFilter('scope', $scope);
        return $result->getIndex($page);
    }

    /**
     * @param string $scope
     * @param string $attrName
     * @param string|array $attrValues
     * @return AssetResfulModel[]|CmsApiIndexIterator
     */
    public function getIndexByAttribute($scope, $attrName, $attrValues)
    {
        $attrValues = (array)$attrValues;
        $result = $this->getBuilder()
            ->setFilter('attr_name', $attrName)
            ->setFilter('attr_values', $attrValues)
            ->setFilter('scope', $scope);
        return $result->getIndex(-1);
    }

    /**
     * @param string $filename
     * @return null|AssetResfulModel
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function fetchItemByFilename($filename)
    {
        $object = $this->fetchItem(null, [
            'filename' => $filename,
        ]);
        return $object instanceof AssetResfulModel ? $object : null;
    }

    /**
     * @param AssetModel $object
     * @return \Cms\Model\Asset\AssetModel|null|\POM\DomainObjectInterface
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function updateItem(AssetModel $object)
    {
        $object = $this->updateObjectFromClient($object, 'asset');
        return $object;
    }

    /**
     * @param AssetModel $object
     * @param null|resource|StreamInterface $filepath
     * @param null|string $filename
     * @param bool $throwOnConflict
     * @return \Cms\Client\Asset\Model\AssetResfulModel|\Cms\Model\Asset\AssetModel
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function createItem(
        AssetModel $object,
        $filepath = null,
        $filename = null,
        bool $throwOnConflict = false
    ) {
        $dataset = $this->convertObjectToArray($object);
        if ($filepath) {
            $multipart = RequestUtils::ConvertArrayToMultipartArray($dataset);
            $multipart[] = [
                'name' => 'file',
                'filename' => $filename,
                'contents' => $filepath,
            ];
            $options = [
                HttpRequestOptions::MULTIPART => true,
                HttpRequestOptions::BODY_CONTENT => $multipart,
            ];
        } else {
            $options = [
                HttpRequestOptions::BODY_CONTENT => $dataset,
            ];
        }

        try {
            // lance la requete vers le server
            $jsonDataset = $this->request('POST', [
                'asset?throw_on_conflict=' . $throwOnConflict,
            ], $options);
        } catch (HttpClientException $e) {
            // l'image existe
            if ($throwOnConflict === true
                && $e->getResponse()->getStatusCode() === 409
            ) {
                $assetId = $e->getMessage();
                $assetModel = $this->fetchItem($assetId);
                if ($assetModel instanceof AssetModel) {
                    return $assetModel;
                }
                throw new \RuntimeException(
                    'cannot fetch the conflict asset ' . $assetId,
                    null,
                    $e
                );
            }
            throw $e;
        }

        if (\is_array($jsonDataset) && !empty($jsonDataset)) {
            $object = $this->withJsonDataset($object, $jsonDataset);
        }
        return $object;
    }
}
