<?php


namespace Cms\Search\Geodata;

use Cms\Model\Geodata\CityModel;
use Cms\Model\Geodata\CityPostalCodeModel;
use Cms\Model\Geodata\PoiModel;
use Cms\Search\GeodataIndexFactory;
use Elastica\Document;
use Move\Search\ElasticMappingBuilder;
use Move\Search\Mapping\ElasticMappingProperty;
use Move\Search\Mapping\KeywordMapping;
use Move\Search\Mapping\ObjectMapping;
use Move\Search\Mapping\TextMapping;

/**
 * Class CityIndexFactory
 * @package Cms\Search\Geodata
 */
class CityIndexFactory
{

    /** Nom du type pour les villes dans l'index */
    public const MAPPING_NAME = 'city';

    /**
     * @return \Move\Search\ElasticMappingBuilder
     */
    public static function createMappingBuilder() : ElasticMappingBuilder
    {
        $builder = new ElasticMappingBuilder(self::MAPPING_NAME, [
            new ElasticMappingProperty('id', 'long'),

            (new KeywordMapping('country_alpha2')),
            (new KeywordMapping('insee_code', false)),

            (new KeywordMapping('admin1_code')),
            (new KeywordMapping('admin2_code', false)),

            (new KeywordMapping('admin1_code_simple')),
            (new KeywordMapping('admin2_code_simple', false)),

            (new KeywordMapping('postal_code'))
                ->withOption('fields', [
                    'raw' => [
                        'type' => 'keyword',
                        'index' => false,
                    ],
                ]),

            (new KeywordMapping('timezone')),
            (new ElasticMappingProperty('geo_location', 'geo_point')),

            (new KeywordMapping('osm_id')),

            (new KeywordMapping('name_locale'))
                ->withOption('fields', [
                    'raw' => [
                        'type' => 'keyword',
                        'index' => false,
                    ],
                ]),

            (new TextMapping('name_fr_fr'))
                ->withAnalyzer(
                    GeodataIndexFactory::createIndexAnalyzer(),
                    GeodataIndexFactory::createSearchAnalyzer()
                )
                ->withOption('fields', [
                    'raw' => [
                        'type' => 'keyword',
                        'index' => false,
                    ],
                ]),

            (new ObjectMapping('poi', [
                (new TextMapping('name_fr_fr'))
                    ->withAnalyzer(
                        GeodataIndexFactory::createIndexAnalyzer(),
                        GeodataIndexFactory::createSearchAnalyzer()
                    )
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                (new KeywordMapping('name_en_en')),
                (new KeywordMapping('categories')),
                (new ElasticMappingProperty('geo_location', 'geo_point')),
                (new KeywordMapping('address')),
                (new KeywordMapping('postal_code'))
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                (new KeywordMapping('city')),
                (new KeywordMapping('insee_code'))
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                (new TextMapping('description_fr_fr'))
                    ->withAnalyzer(
                        GeodataIndexFactory::createIndexAnalyzer(),
                        GeodataIndexFactory::createSearchAnalyzer()
                    )
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                (new TextMapping('description_en_en'))
                    ->withOption('fields', [
                        'raw' => [
                            'type' => 'keyword',
                            'index' => false,
                        ],
                    ]),
                (new TextMapping('uri')),
                (new ElasticMappingProperty('last_update_datatourisme', 'date'))
            ]))
        ]);


        return $builder;
    }

    /**
     * @param CityModel $cityModel
     * @param \Cms\Model\Geodata\CityPostalCodeModel[] $postalCodes
     * @param \Cms\Model\Geodata\PoiModel[] $pois
     * @return \Elastica\Document
     * @throws \InvalidArgumentException
     */
    public static function createDocument(CityModel $cityModel, array $postalCodes = [], array $pois = []) : Document
    {
        if (!$cityModel instanceof CityModel) {
            throw new \InvalidArgumentException('can only create document from CityModel');
        }

        // récuperation des code postaux
        $postalCode = [];
        foreach ($postalCodes as $postalCodeModel) {
            if ($postalCodeModel instanceof CityPostalCodeModel) {
                $postalCode[] = $postalCodeModel->postal_code;
            }
        }

        // data du document
        $cityData = [
            'country_alpha2' => $cityModel->country_alpha2,
            'insee_code' => $cityModel->insee_code,
            'admin1_code' => $cityModel->admin1_code,
            'admin2_code' => $cityModel->admin2_code,
            'admin1_code_simple' => $cityModel->getAdminCodeSimple(1),
            'admin2_code_simple' => $cityModel->getAdminCodeSimple(2),
            'timezone' => $cityModel->timezone,
            'geo_location' => $cityModel->gps_location ? [
                'lat' => $cityModel->gps_location->latitude,
                'lon' => $cityModel->gps_location->longitude,
            ] : null,
            'osm_id' => $cityModel->osm_id,
            'name_locale' => $cityModel->name,
            'name_fr_fr' => $cityModel->name,
            'postal_code' => $postalCode ?: null,
            'poi' => [],
        ];

        if (!empty($pois)) {
            /** @var PoiModel $poi */
            foreach ($pois as $poi) {
                $name = json_decode($poi->name, true);
                $cityData['poi'][] = [
                    'name_fr_fr' => $name['fr'] ?? '',
                    'name_en_en' => $name['en'] ?? '',
                    'categories' => $poi->categories,
                    'geo_location' => $poi->gps_location ? [
                        'lat' => $poi->gps_location->latitude,
                        'lon' => $poi->gps_location->longitude,
                    ] : null,
                    'address' => $poi->address,
                    'postal_code' => $poi->postal_code,
                    'city' => $poi->city,
                    'insee_code' => $poi->insee_code,
                    'description_fr_fr' => $poi->description['fr'] ?? '',
                    'description_en_en' => $poi->description['en'] ?? '',
                    'uri' => $poi->uri,
                    'last_update_datatourisme' => $poi->last_update_datatourisme->format('Y-m-d\TH:i:s\Z'),
                ];
            }
        }

        // envoi des données
        $document = new Document($cityModel->id, $cityData, self::MAPPING_NAME);
        return $document;
    }
}
