<?php

namespace Cms\Client\Campsite\Helper;

use Cms\Client\AddressBook\Model\AddressRestfulModel;
use Cms\Client\Campsite\Model\CampsiteRestfulModel;
use Cms\Client\Campsite\Model\ThematicRestfulModel;
use Cms\Client\Command\Campsite\Command\FetchBulkCampsiteCommand;
use Cms\Client\Command\Campsite\Command\FetchBulkProductCommand;
use Cms\Client\Command\CommandExecutorHelper;
use Cms\Client\Command\ExceptionDecoratorInterface;
use Cms\Client\Service\CampsiteService;
use Cms\Command\FetchWithScopeCommand;
use Cms\Model\AddressBook\Contact\ContactType;
use Move\Command\Exception\CommandException;
use Move\Command\Exception\NotFoundException;

/**
 * Trait CampsiteHelperTrait
 * @package Cms\Client\Campsite\Helper
 */
trait CampsiteHelperTrait
{

    /**
     * @return \Cms\Client\Service\CampsiteService
     * @throws \DomainException
     */
    abstract public function getCampsiteService() : CampsiteService;

    /**
     * @param \Cms\Client\AddressBook\Model\AddressRestfulModel $addressModel
     * @return null|string
     */
    protected function getCampsiteEmail(AddressRestfulModel $addressModel)
    {
        // récupération de l'adresse administrative
        $email = null;
        // recuperation de l'email
        foreach ($addressModel->contacts as $contact) {
            if ((string)$contact->type === ContactType::EMAIL) {
                $email = $contact->value;
                break;
            }
        }
        return $email;
    }

    /**
     * @param string|int $identifier Thematic slug or id
     * @param int|null $scopeId obligatoire si slug comme $identifier
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Campsite\Model\ThematicRestfulModel
     * @throws \DomainException
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getThematicModel(
        $identifier,
        int $scopeId = null,
        ExceptionDecoratorInterface $decorator = null
    ) : ThematicRestfulModel {
        $commandBus = $this->getCampsiteService()->getThematicBus();
        $command = new FetchWithScopeCommand($identifier, $scopeId);

        // execution
        $thematicModel = CommandExecutorHelper::handleCmd(
            $commandBus,
            $command,
            $decorator
        );
        return $thematicModel;
    }

    /**
     * @param array $codes
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return array
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getProductCodeMap(
        array $codes,
        ExceptionDecoratorInterface $decorator = null
    ) : array {
        // récuperation des id de produit
        $productCodeMap = [];
        try {
            $productBus = $this->getCampsiteService()->getProductBus();
            $command = new FetchBulkProductCommand($codes);
            $products = $productBus->handle($command);
            foreach ($products as $product) {
                $productCodeMap[$product->code] = $product->id;
            }
        } catch (NotFoundException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getNotFoundDecorator($e));
            }
            throw $e;
        } catch (CommandException $e) {
            if ($decorator !== null) {
                \call_user_func($decorator->getThrowableDecorator($e));
            }
            throw $e;
        }
        return $productCodeMap;
    }

    /**
     * @param array $identifiers
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return mixed
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getBulkCampsiteModel(
        array $identifiers,
        ExceptionDecoratorInterface $decorator = null
    ) {
        // récuperation des campings
        $commandBus = $this->getCampsiteService()->getCampsiteBus();
        $command = new FetchBulkCampsiteCommand($identifiers);
        // execution
        $campsiteModels = CommandExecutorHelper::handleCmd(
            $commandBus,
            $command,
            $decorator
        );
        return $campsiteModels;
    }

    /**
     * @param string|int $identifier Campsite slug or id
     * @param int|null $scopeId obligatoire si slug comme $identifier
     * @param \Cms\Client\Command\ExceptionDecoratorInterface|null $decorator
     * @return \Cms\Client\Campsite\Model\CampsiteRestfulModel
     * @throws \DomainException
     * @throws \Move\Command\Exception\CommandException
     */
    protected function getCampsiteModel(
        $identifier,
        int $scopeId = null,
        ExceptionDecoratorInterface $decorator = null
    ) : CampsiteRestfulModel {
        $commandBus = $this->getCampsiteService()->getCampsiteBus();
        $command = new FetchWithScopeCommand($identifier, $scopeId);
        // execution
        $campsiteModel = CommandExecutorHelper::handleCmd(
            $commandBus,
            $command,
            $decorator
        );
        return $campsiteModel;
    }

    /**
     * @param \Cms\Client\Campsite\Model\CampsiteRestfulModel $model
     * @param callable|null $transformer surcharge le transformer de base
     * @return array
     * @throws \DomainException
     */
    protected function campsiteToArray(
        CampsiteRestfulModel $model,
        callable $transformer = null
    ) : array {
        $transformer = $transformer ?: $this->getCampsiteService()->getCampsiteTransformer();
        $dataset = $transformer($model);
        if (!\is_array($dataset)) {
            throw new \DomainException('transformer must return an array');
        }
        return $dataset;
    }

    /**
     * @param string|int $identifier Campsite slug or id
     * @param int|null $scopeId obligatoire si slug comme $identifier
     * @param callable|null $transformer surcharge le transformer de base
     * @return array
     * @throws \Move\Command\Exception\CommandException
     * @throws \DomainException
     */
    protected function getCampsiteArray(
        $identifier,
        int $scopeId = null,
        callable $transformer = null
    ) : array {
        $campsiteModel = $this->getCampsiteModel($identifier, $scopeId);
        return $this->campsiteToArray($campsiteModel, $transformer);
    }
}
