<?php

namespace Move\I18n;

use Move\Utils\Str;

/**
 * @param string $lang
 * @param string $locale
 * @return string|null
 */
function languageCodeToName(string $lang, $locale = 'fr')
{
    static $languageData = null;
    if ($languageData === null) {
        $jsonContent = file_get_contents(__DIR__ . '/../../../resources/iso/iso_639-2.json');
        $languageData = Str::parseJson($jsonContent);
    }
    if (!empty($languageData[$lang][$locale])) {
        return $languageData[$lang][$locale][0];
    }
    return null;
}

/**
 * @param int $category
 * @param string $fallback
 * @return string
 */
function getlocale(int $category = LC_ALL, string $fallback = 'en_US.UTF-8')
{
    $config = explode(';', \setlocale($category, 0));
    if (!empty($config[0])) {
        return $config[0];
    }
    return $fallback;
}

/**
 * @param array|string $local
 * @return string
 */
function setlocale($local)
{
    // recherche de la local actuel
    $current = getlocale();
    if (\is_string($local)) {
        putenv('LANG=' . $local);
        putenv('LC_ALL=' . $local);
    }
    $local = (array)$local;
    if (!\setlocale(LC_ALL, ...$local)) {
        throw new \RuntimeException('locale ' . implode(', ', $local) . ' not installed on system');
    }
    \setlocale(LC_TIME, ...$local);
    return $current;
}

/**
 * @param callable $function
 * @param array|string $local
 * @param array $args
 * @return mixed
 */
function callInLocale(callable $function, $local, array $args = [])
{
    $current = setlocale($local);
    $result = $function(...$args);
    setlocale($current);
    return $result;
}

/**
 * @param int|null $category
 * @param string|null $fallbackLang
 * @return string|null
 */
function detectLngWithLocale(int $category = null, string $fallbackLang = null)
{
    try {
        $currentLocale = getlocale($category ?: LC_ALL);
        return localeCodeToLng($currentLocale, $fallbackLang);
    } catch (\Exception $e) {
    }
    return $fallbackLang;
}

/**
 * @param string $locale
 * @param string|null $fallbackLang
 * @return string
 * @throws \InvalidArgumentException
 */
function localeCodeToLng(string $locale, string $fallbackLang = null) : string
{
    if (1 === preg_match('@^([a-z]{2,3})(?:-|_)([a-z]{2})@i', $locale, $matches)) {
        return countryCodeToLng($matches[1], $fallbackLang);
    }
    if ($fallbackLang !== null) {
        return $fallbackLang;
    }
    throw new \InvalidArgumentException('locale non prise en charge : ' . $locale);
}

/**
 * @param string $countryCode
 * @param string|null $fallbackLang
 * @return string
 * @throws \InvalidArgumentException
 */
function countryCodeToLng(
    string $countryCode = null,
    string $fallbackLang = null
) : string {
    if (empty($countryCode)) {
        if ($fallbackLang !== null) {
            return $fallbackLang;
        }
        throw new \InvalidArgumentException('countryCode non prise en charge : ' . $countryCode);
    }
    $countryCode = strtolower($countryCode);
    if (I18nLanguageCode::isValid($countryCode)) {
        return $countryCode;
    }
    switch ($countryCode) {
        case 'fr':
            return I18nLanguageCode::FR;
            break;
        case 'uk':
        case 'gb':
        case 'en':
            return I18nLanguageCode::EN;
            break;
        case 'it':
            return I18nLanguageCode::IT;
            break;
        case 'es':
            return I18nLanguageCode::ES;
            break;
        case 'de':
            return I18nLanguageCode::DE;
            break;
        case 'nl':
            return I18nLanguageCode::NL;
            break;
        default:
            if ($fallbackLang !== null) {
                return $fallbackLang;
            }
            throw new \InvalidArgumentException('countryCode non prise en charge : ' . $countryCode);
            break;
    }
}
