<?php

namespace Cms\Client\Command\Campsite\Thematic;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Campsite\Model\ThematicRestfulModel;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Service\CampsiteService;
use Cms\Command\FetchWithScopeCommand;
use Move\Cache\CacheAwareInterface;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\NotFoundException;
use Move\Http\Client\Exception\HttpRequestException;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class ThematicCommandHandler
 * @package Cms\Client\Command\Campsite\Thematic
 */
class ThematicCommandHandler implements
    LoggerAwareInterface,
    CacheAwareInterface
{
    use CacheHelperTrait;
    use LoggerAwareTrait;
    use CommandExceptionHandleTrait;

    /**
     * @var CampsiteService
     */
    protected $service;

    /**
     * ThematicCommandHandler constructor.
     * @param CampsiteService $service
     */
    public function __construct(CampsiteService $service)
    {
        $this->service = $service;
    }

    /**
     * @param \Cms\Command\FetchWithScopeCommand $command
     * @return \Cms\Client\Campsite\Model\ThematicRestfulModel
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\NotFoundException
     * @throws \RuntimeException
     */
    public function handleFetchWithScope(FetchWithScopeCommand $command) : ThematicRestfulModel
    {
        $identifier = $command->getId();
        if (!$identifier) {
            throw new BadParameterException('identifier is empty');
        }
        $scopeId = $command->getScopeId();
        if (!is_numeric($identifier) && (!$identifier xor !$scopeId)) {
            throw new BadParameterException('scopeid must be define if identifier is not numeric');
        }

        $thematicClient = $this->service->getThematicClient();
        try {
            if (is_numeric($identifier)) {
                // récuperation du camping par slug
                $thematicModel = $thematicClient->fetchItem(
                    $identifier
                );
            } else {
                // récuperation du camping par slug
                $thematicModel = $thematicClient->fetchItemBySlug(
                    $scopeId,
                    $identifier
                );
            }
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('CampsiteCommandHandler', $e);
            $thematicModel = null;
        }

        if (!$thematicModel instanceof ThematicRestfulModel) {
            throw new NotFoundException(
                'thematic with identifier ' . $identifier . ' does not exist'
            );
        }
        return $thematicModel;
    }
}
