<?php

namespace Cms\LinkResource;

use Cms\LinkResource\Provider\DataAdapterAwareInterface;
use Cms\LinkResource\Provider\ProviderAdapterInterface;
use Cms\LinkResource\Provider\RecursiveAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareInterface;
use Move\Specification\EnumSpecification;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Class LinkResourceHandler
 * @package Cms\LinkResource
 */
class LinkResourceHandler implements
    LoggerAwareInterface
{
    use LoggerAwareTrait;

    /** @var  \CallbackFilterIterator|ProviderAdapterInterface[] */
    protected $providerIterator;

    /**
     * LinkResourceHandler constructor.
     * @param ProviderAdapterInterface[]|array $providerAdapters
     */
    public function __construct(array $providerAdapters)
    {
        $providerIterator = new \ArrayIterator($providerAdapters);
        $this->providerIterator = new \CallbackFilterIterator(
            $providerIterator,
            function ($current) {
                return $current instanceof ProviderAdapterInterface;
            }
        );
        $this->logger = new NullLogger();
    }

    /**
     * @param LinkResourceModel $resourceModel
     * @return ProviderAdapterInterface|null
     */
    public function getAdapterByModel(LinkResourceModel $resourceModel)
    {
        return $this->getAdapterByProviderCode($resourceModel->link_provider);
    }

    /**
     * @param string $providerCode
     * @return ProviderAdapterInterface|null
     */
    public function getAdapterByProviderCode($providerCode)
    {
        foreach ($this->providerIterator as $provider) {
            if ($provider->hasName($providerCode)) {
                return $provider;
            }
        }
        return null;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @return \CallbackFilterIterator|ProviderAdapterInterface[]
     * @throws \InvalidArgumentException
     */
    public function getAdaptersByTypeCode($typeCode)
    {
        if (!(new EnumSpecification(LinkResourceType::class))->isSatisfiedBy($typeCode)) {
            throw new \InvalidArgumentException('resource type is invalid');
        }
        $providers = new \CallbackFilterIterator(
            $this->providerIterator,
            function ($current) use ($typeCode) {
                return $current instanceof ProviderAdapterInterface
                    && $current->hasType($typeCode);
            }
        );
        return $providers;
    }

    /**
     * @param \Cms\LinkResource\LinkResourceModel $resourceModel
     * @return \Cms\LinkResource\LinkResourceCollection
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getResourcesByModel(LinkResourceModel $resourceModel) : LinkResourceCollection
    {
        return $this->getResources(
            $resourceModel->link_provider,
            $resourceModel->link_type,
            $resourceModel->link_ref
        );
    }

    /**
     * @param string $providerCode
     * @param LinkResourceType|string $typeCode
     * @param string $objectId
     * @return \Cms\LinkResource\LinkResourceCollection
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getResources(
        string $providerCode,
        $typeCode,
        $objectId
    ) : LinkResourceCollection {
        if (!(new EnumSpecification(LinkResourceType::class))->isSatisfiedBy($typeCode)) {
            throw new \InvalidArgumentException('resource type is invalid');
        }
        $client = $this->getAdapterByProviderCode($providerCode);
        if (!$client) {
            throw new \UnexpectedValueException('no client for provder ' . $providerCode);
        }
        if (!$client instanceof RecursiveAdapterAwareInterface) {
            throw new \UnexpectedValueException(
                'recursive client for provider ' . $providerCode . ' must implement RecursiveAdapterAwareInterface'
            );
        }
        $resources = $client->getResources($typeCode, $objectId);
        $resourceCol = new LinkResourceCollection($resources, true);
        return $resourceCol;
    }

    /**
     * @param LinkResourceModel $resourceModel
     * @return mixed
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getDataByModel(LinkResourceModel $resourceModel)
    {
        return $this->getData(
            $resourceModel->link_provider,
            $resourceModel->link_type,
            $resourceModel->link_ref
        );
    }

    /**
     * @param string $providerCode
     * @param LinkResourceType|string $typeCode
     * @param string $objectId
     * @return mixed
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getData($providerCode, $typeCode, $objectId)
    {
        if (!(new EnumSpecification(LinkResourceType::class))->isSatisfiedBy($typeCode)) {
            throw new \InvalidArgumentException('resource type is invalid');
        }
        $client = $this->getAdapterByProviderCode($providerCode);
        if (!$client) {
            throw new \UnexpectedValueException('no client for provder ' . $providerCode);
        }
        if (!$client instanceof DataAdapterAwareInterface) {
            throw new \UnexpectedValueException(
                'dataClient for provider ' . $providerCode . ' must implement DataAdapterAwareInterface'
            );
        }
        return $client->getData($typeCode, $objectId);
    }


    /**
     * @param LinkResourceModel $resourceModel
     * @return mixed
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getLabelByModel(LinkResourceModel $resourceModel)
    {
        return $this->getLabel(
            $resourceModel->link_provider,
            $resourceModel->link_type,
            $resourceModel->link_ref
        );
    }

    /**
     * @param string $providerCode
     * @param LinkResourceType|string $typeCode
     * @param string $objectId
     * @return mixed
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getLabel($providerCode, $typeCode, $objectId)
    {
        if (!(new EnumSpecification(LinkResourceType::class))->isSatisfiedBy($typeCode)) {
            throw new \InvalidArgumentException('resource type is invalid');
        }
        $client = $this->getAdapterByProviderCode($providerCode);
        if (!$client instanceof DataAdapterAwareInterface) {
            throw new \UnexpectedValueException('dataClient must implement DataAdapterAwareInterface');
        }
        return $client->getLabel($typeCode, $objectId);
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param string $query
     * @return array
     * @throws \InvalidArgumentException
     */
    public function search($typeCode, $query)
    {
        // On récupère les providers qui correspondent au type
        $providers = $this->getAdaptersByTypeCode($typeCode);
        // On récupère le résultat pour chaque provider qui correspond
        $results = [];
        foreach ($providers as $provider) {
            if ($provider instanceof SearchAdapterAwareInterface) {
                try {
                    $results[$provider->getIdentifier()] = $provider->search($typeCode, $query);
                } catch (\Exception $e) {
                    $this->logger->alert(
                        'failed in search provider ' . $provider->getIdentifier(),
                        ['exception' => $e]
                    );
                }
            }
        }

        return $results;
    }
}
