<?php


namespace Cms\LinkResource\Provider\CMS;

use Cms\Client\Campsite\Model\CampsiteRestfulModel;
use Cms\Client\Service\CampsiteService;
use Cms\LinkResource\LinkResourcePrefsIdentifier;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\ProviderClientAdapterInterface;
use Cms\LinkResource\Provider\ProviderRestfulClientAdapter;
use Cms\LinkResource\Provider\SearchAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareTrait;
use Cms\Model\AddressBook\Address\AddressModel;
use Cms\Search\Campsite\CampsiteSearchClient;
use Cms\Search\Campsite\CampsiteSearchQueryBuilder;
use Cms\Search\Provider\Elastica\ElasticSearchResultElement;
use Move\Iterator\MapIterator;
use POM\DomainObjectInterface;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Log\LoggerInterface;

/**
 * Class CmsCampsiteProvider
 * @package Cms\LinkResource\Provider\CMS
 */
class CmsCampsiteProvider extends AbstractCmsCampsiteProvider implements
    SearchAdapterAwareInterface
{
    use SearchAdapterAwareTrait;

    /** @var int  */
    public static $SCOPE_CAMPSITE = 6;

    /** @var \Cms\Client\Service\CampsiteService */
    private $campsiteService;

    /** @var \Cms\Search\Campsite\CampsiteSearchClient|null */
    private $searchClient;

    /** @var \Cms\LinkResource\Provider\CMS\AbstractCmsAddressBookProvider|null */
    private $bookProvider;

    /**
     * CmsCampsiteProvider constructor.
     * @param \Cms\Client\Service\CampsiteService $campsiteService
     * @param \Cms\Search\Campsite\CampsiteSearchClient|null $searchClient
     * @param \Cms\LinkResource\Provider\CMS\AbstractCmsAddressBookProvider|null $bookProvider
     * @param \Psr\Cache\CacheItemPoolInterface $cachePool
     * @param \Psr\Log\LoggerInterface|null $logger
     */
    public function __construct(
        CampsiteService $campsiteService,
        CampsiteSearchClient $searchClient = null,
        AbstractCmsAddressBookProvider $bookProvider = null,
        CacheItemPoolInterface $cachePool = null,
        LoggerInterface $logger = null
    ) {
        parent::__construct($cachePool, $logger);
        $this->campsiteService = $campsiteService;
        $this->searchClient = $searchClient;
        $this->bookProvider = $bookProvider;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @return \Cms\Client\Campsite\CampsiteClient|\Cms\Client\Campsite\GroupClient|\Cms\Client\Campsite\ThematicClient|null
     */
    protected function getClientByTypeCode($typeCode)
    {
        switch ((string)$typeCode) {
            case LinkResourceType::CAMPSITE:
                return $this->campsiteService->getCampsiteClient();
                break;
            case LinkResourceType::CAMPSITE_THEMATIC:
                return $this->campsiteService->getThematicClient();
                break;
            case LinkResourceType::CAMPSITE_GROUP:
                return $this->campsiteService->getGroupClient();
                break;
        }
        return null;
    }

    /**
     * @param \POM\DomainObjectInterface $model
     * @return null|string
     */
    public function getLabelFromModel(DomainObjectInterface $model)
    {
        $label = parent::getLabelFromModel($model);
        if ($model instanceof CampsiteRestfulModel) {
            // ajout du label de l'address
            $resource = $model->getFirstAddress(LinkResourcePrefsIdentifier::ADDRESS_LOC);
            if ($resource !== null
                && ($data = $this->bookProvider->getData($resource->link_type, $resource->link_ref))
                && $data instanceof AddressModel
            ) {
                $label .= sprintf(' (%s - %s)', $data->city_name, $data->getAdmin2Code());
            }
        }
        return $label;
    }


    /**
     * @param mixed $typeCode
     * @return \Cms\LinkResource\Provider\ProviderClientAdapterInterface
     * @throws \RuntimeException
     */
    protected function getClientAdapter($typeCode) : ProviderClientAdapterInterface
    {
        $client = $this->getClientByTypeCode($typeCode);
        if ($client !== null) {
            return new ProviderRestfulClientAdapter($client);
        }
        throw new \RuntimeException('no client adapter');
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param mixed $query
     * @return array
     * @throws \InvalidArgumentException
     */
    public function execSearch($typeCode, $query) : array
    {
        // recherche par id
        if (is_numeric($query)) {
            $resource = $this->getData($typeCode, $query);
            if ($resource) {
                return [
                    $resource->id,
                ];
            }
        }
        $country = null;
        if (\is_array($query)) {
            $country = $query['country'] ?? null;
            $query = $query['text'] ?? '';
        }
        if (empty($query)) {
            return [];
        }
        switch ((string)$typeCode) {
            case LinkResourceType::CAMPSITE:
                if (!$this->searchClient) {
                    return [];
                }
                $builder = new CampsiteSearchQueryBuilder(self::$SCOPE_CAMPSITE);
                $builder
                    ->setSearchTerm($query)
                    ->setSourceFilter(['id']);
                if (!empty($country)) {
                    $builder->setCountryCode($country);
                }
                $results = $this->searchClient->search($builder);
                $results = new MapIterator(
                    $results,
                    function (ElasticSearchResultElement $element) {
                        return $element->getId();
                    }
                );
                return iterator_to_array($results, false);
                break;
            case LinkResourceType::CAMPSITE_GROUP:
            case LinkResourceType::CAMPSITE_THEMATIC:
                if (!$this->scopeId) {
                    return [];
                }
                $results = $this->getClientByTypeCode($typeCode)
                    ->getIndexBySearch($this->scopeId, $query);
                if (empty($results)) {
                    return [];
                }
                // On récupère les id de chaque modele pour les renvoyer
                $results = iterator_to_array($results, false);
                foreach ($results as &$model) {
                    $model = $model->id;
                }
                return $results;
                break;
        }
        return [];
    }
}
