<?php


namespace Move\Search\Mapping;

use Move\Search\Analyzer\ElasticAnalyzer;

/**
 * Class ElasticMappingProperty
 * @package Move\Search\Mapping
 */
class ElasticMappingProperty
{

    /** @var array */
    private $opts;

    /** @var string */
    private $propName;

    /**
     * ElasticMappingProperty constructor.
     * @param string $propName
     * @param string $type
     * @param bool|null $includeInAll
     * @param array $opts
     */
    public function __construct(
        string $propName,
        string $type,
        $includeInAll = true,
        array $opts = []
    ) {
        $this->propName = $propName;
        $this->opts = $opts;
        $this->opts['type'] = $type;
        /*if (null !== $includeInAll) {
            $this->opts['include_in_all'] = (bool)$includeInAll;
        }*/
    }

    /**
     * @return array
     */
    public function toArray() : array
    {
        return $this->opts;
    }

    /**
     * @param \Move\Search\Analyzer\ElasticAnalyzer|null $analyzer
     * @param \Move\Search\Analyzer\ElasticAnalyzer|null $searchAnalyzer
     * @return \Move\Search\Mapping\ElasticMappingProperty
     */
    public function withAnalyzer(
        ElasticAnalyzer $analyzer = null,
        ElasticAnalyzer $searchAnalyzer = null
    ) : ElasticMappingProperty {
        $clone = $this;
        if ($searchAnalyzer !== null) {
            $clone = $this->withOption('search_analyzer', $searchAnalyzer->getName());
        }
        if ($analyzer !== null) {
            $clone = $this->withOption('analyzer', $analyzer->getName());
        }
        return $clone;
    }

    /**
     * @param string $name
     * @param mixed $value
     * @return \Move\Search\Mapping\ElasticMappingProperty
     */
    public function withOption(string $name, $value) : ElasticMappingProperty
    {
        $clone = clone $this;
        $clone->opts[$name] = $value;
        return $clone;
    }

    /**
     * @param float|null $boost
     * @return \Move\Search\Mapping\ElasticMappingProperty
     */
    public function withBoost(float $boost) : ElasticMappingProperty
    {
        return $this->withOption('boost', $boost);
    }

    /**
     * @param string $propName
     * @return ElasticMappingProperty
     */
    public function withPropName(string $propName) : ElasticMappingProperty
    {
        $clone = clone $this;
        $clone->propName = $propName;
        return $this;
    }

    /**
     * @param bool $analyzed
     * @return ElasticMappingProperty
     */
    public function withAnalyzed(bool $analyzed) : ElasticMappingProperty
    {
        $clone = clone $this;
        if ($analyzed === false) {
            $clone->opts['index'] = false;
        } else {
            unset($clone->opts['index']);
        }
        return $clone;
    }

    /**
     * @return string
     */
    public function getPropName() : string
    {
        return $this->propName;
    }
}
