<?php

namespace Move\Http;

use Interop\Container\ContainerInterface;
use League\Route\Http\Exception;
use League\Route\RouteCollection;
use Move\Http\Middleware\BodyParser;
use Move\Http\Middleware\MiddlewareInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;
use Psr\Log\NullLogger;
use Zend\Diactoros\Response\SapiEmitter;

/**
 * Class Application
 * @package Move\Http
 */
class Application implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    /** @var string */
    private $appName;

    /**
     * @var ContainerInterface
     */
    private $container;

    /**
     * @var ServerRequestInterface
     */
    private $request;

    /**
     * @var ResponseInterface
     */
    private $response;

    /**
     * @var MiddlewareInterface[]
     */
    private $middleware;

    /**
     * Application constructor.
     * @param string $appName
     * @param ContainerInterface $container
     * @throws \Psr\Container\ContainerExceptionInterface
     * @throws \Psr\Container\NotFoundExceptionInterface
     */
    public function __construct($appName, ContainerInterface $container)
    {
        $this->appName = $appName;
        $this->container = $container;

        //
        if (!$container->has(ServerRequestInterface::class)
            || !$container->has(ResponseInterface::class)
        ) {
            throw new \InvalidArgumentException('ServerRequestInterface & ResponseInterface must be defined in container');
        }

        // inject des dependances
        $this
            ->setRequest($container->get(ServerRequestInterface::class))
            ->setResponse($container->get(ResponseInterface::class));

        // inject logger
        $this->logger = new NullLogger();
        if ($container->has(LoggerInterface::class)) {
            $this->logger = $container->get(LoggerInterface::class);
        }

        // inject les middleware de request
        $this->middleware[] = new BodyParser();
    }


    /**
     * @param ServerRequestInterface $request
     * @return $this
     */
    public function setRequest(ServerRequestInterface $request)
    {
        $this->request = $request;
        return $this;
    }

    /**
     * @param ResponseInterface $response
     * @return $this
     */
    public function setResponse(ResponseInterface $response)
    {
        $this->response = $response;
        return $this;
    }

    /**
     * @param BootstrapServerInterface $bootstrap
     * @throws \InvalidArgumentException
     */
    public function process(BootstrapServerInterface $bootstrap)
    {
        // demarrage du routage
        $router = new RouteCollection($this->container);

        // Demarrage de l'application
        $bootstrap->handleServer($router);

        // lance les middlewares sur la request/response
        $request = $this->request;
        foreach ($this->middleware as $middleware) {
            $result = $middleware->handle($request);
            if ($result instanceof ResponseInterface) {
                // send response
                (new SapiEmitter())->emit($result);
                return;
            }
            if ($result instanceof ServerRequestInterface) {
                $request = $result;
            }
        }

        // recup de la reponse predefini
        $response = $this->response;

        // routage
        try {
            // dispatch
            $response = $router->dispatch($request, $response);

            // contenu vide
            if (200 === (int)$response->getStatusCode() && !$response->getBody()->getSize()) {
                $response = $response->withStatus(204);
            }
        } catch (Exception $e) {
            // traitement de l'erreur de routage
            $response = $response->withStatus($e->getStatusCode());

            foreach ($e->getHeaders() as $headerName => $headerStr) {
                $response = $response->withAddedHeader($headerName, $headerStr);
            }

            // on ne logg que les erreur serveur
            if ($e->getStatusCode() >= 500) {
                $this->logException($e, LogLevel::ERROR, $request, $response);
            } else {
                $this->logException($e, LogLevel::DEBUG, $request, $response);
            }
        } catch (\Exception $e) {
            $response = $response->withStatus(500);
            $this->logException($e, LogLevel::CRITICAL, $request, $response);
        } catch (\Throwable $e) {
            $response = $response->withStatus(500);
            $this->logException($e, LogLevel::ALERT, $request, $response);
        }

        // send response
        (new SapiEmitter())->emit($response);
    }

    /**
     * @param \Throwable $e
     * @param mixed $level
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     */
    private function logException(
        \Throwable $e,
        $level,
        ServerRequestInterface $request,
        ResponseInterface $response
    ) {
        $type = 'AppError';
        $this->logger->log($level, $type . ' "' . $e->getMessage() . '"', [
            'request_uri' => (string)$request->getUri(),
            'request_meth' => $request->getMethod(),
            'response_code' => $response->getStatusCode(),
            'response_reason' => $response->getReasonPhrase(),
            'err_message' => $e->getMessage(),
            'err_class' => \get_class($e),
            'err_type' => $type,
            'exception' => $e,
        ]);
    }

    /**
     * @param Middleware\MiddlewareInterface $middleware
     * @return Application
     */
    public function addMiddleware(MiddlewareInterface $middleware)
    {
        $this->middleware[] = $middleware;
        return $this;
    }
}
