<?php

namespace Move\Http\Strategy\DataContainer;

use PHPUnit\Framework\TestCase;

/**
 * Class DataContainerTest
 * @package Move\Http\Strategy\DataContainer
 */
class DataContainerTest extends TestCase
{
    protected $dataset = [
        'data'=> [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]
    ];

    /**
     * @covers DataContainer::__construct
     * @return DataContainer
     */
    public function testConstruct()
    {
        $instance = new DataContainer($this->dataset);
        $this->assertInstanceOf(DataContainer::class, $instance);
        return $instance;
    }

    /**
     * @depends testConstruct
     * @covers DataContainer::offsetExists
     * @param DataContainer $container
     */
    public function testOffsetExists(DataContainer $container)
    {
        $this->assertTrue($container->offsetExists('data'));

        $this->assertFalse($container->offsetExists('inexistant'));
    }

    /**
     * @depends testConstruct
     * @covers DataContainer::offsetGet
     * @param DataContainer $container
     * @return DataContainer
     */
    public function testOffsetGet(DataContainer $container)
    {
        $empty_container = new DataContainer();
        $offest = 'data';

        $data = $empty_container->offsetGet($offest);
        $this->assertEmpty($data);

        $data = $container->offsetGet($offest);
        $this->assertNotEmpty($data);

        $this->assertNull($container->offsetGet('inexistant'));

        return $container;
    }

    /**
     * @depends testOffsetGet
     * @covers DataContainer::offsetSet
     */
    public function testOffsetSet()
    {
        $container = new DataContainer();
        $offest = 'data';
        $newData = range(10, 20);

        $container->offsetSet($offest, $newData);
        $this->assertEquals($newData, $container->offsetGet($offest));

        $container->offsetSet('inexistant', [1,2,3]);
        $this->assertEquals($newData, $container->offsetGet($offest));
    }

    /**
     * @depends testConstruct
     * @covers DataContainer::offsetUnset
     * @param DataContainer $container
     *
     * @expectedException \BadMethodCallException
     */
    public function testOffsetUnset(DataContainer $container)
    {
        $container->offsetUnset('data');
    }

    /**
     * @depends testConstruct
     * @covers DataContainer::getDataset
     * @param DataContainer $container
     *
     * @return DataContainer
     */
    public function testGetDataset(DataContainer $container)
    {
        $dataset = $container->getDataset();
        $this->assertEquals($this->dataset, $dataset);

        return $container;
    }
    /**
     * @depends testGetDataset
     * @covers DataContainer::setDataset
     * @param DataContainer $container
     */
    public function testSetDataset(DataContainer $container)
    {
        $newDataset = \json_decode('{"attr":"lorem ipsum"}');
        $container->setDataset($newDataset);

        $this->assertEquals($newDataset, $container->getDataset());
    }

    /**
     * @depends testGetDataset
     * @covers DataContainer::jsonSerialize
     * @param DataContainer $container
     */
    public function testJsonSerialize(DataContainer $container)
    {
        $serializable = $container->jsonSerialize();
        $this->assertNotInternalType('resource', $serializable);
    }
}
