<?php

namespace Cms\Filesystem\Strategy;

use League\Flysystem\Filesystem;

/**
 * Class Md5RenameStrategy
 * @package Cms\Filesystem\Strategy
 */
class Md5RenameStrategy implements RenameStrategyInterface
{

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * Md5RenameStrategy constructor.
     * @param Filesystem $filesystem
     */
    public function __construct(Filesystem $filesystem)
    {
        $this->filesystem = $filesystem;
    }

    /**
     * @param string $filePath
     * @param bool $override
     * @return string
     * @throws \UnexpectedValueException
     * @throws \League\Flysystem\FileNotFoundException
     * @throws \League\Flysystem\FileExistsException
     * @throws \InvalidArgumentException
     */
    public function __invoke($filePath, $override = false)
    {
        return $this->rename($filePath);
    }

    /**
     * @param string $filename
     * @return string
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    protected function createFilename($filename) : string
    {
        // check du filename
        if (!is_file($filename)) {
            throw new \InvalidArgumentException('file does not exist');
        }
        // fabrication du nouveau filename
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if (empty($ext)) {
            throw new \UnexpectedValueException('extension not found');
        }
        // changement du nom de fichier
        return md5_file($filename) . '.' . $ext;
    }

    /**
     * @param string $filename
     * @return string
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function getDestinationBasename($filename) : string
    {
        return $this->createFilename($filename);
    }

    /**
     * @param string $filename
     * @return string
     * @throws \UnexpectedValueException
     */
    protected function createDirectory($filename) : string
    {
        // creation du repertoire
        $directory = $this->getDestinationPath($filename);
        if (!$this->filesystem->has($directory) && !$this->filesystem->createDir($directory)) {
            throw new \UnexpectedValueException("directory $directory cannot be created");
        }
        return $directory;
    }

    /**
     * @param string $oldPath
     * @param string $newPath
     * @param bool $override
     * @return bool
     * @throws \League\Flysystem\FileNotFoundException
     * @throws \League\Flysystem\FileExistsException
     * @throws \InvalidArgumentException
     */
    protected function execRename($oldPath, $newPath, $override = false) : bool
    {
        if ($override === true && $this->filesystem->has($newPath)) {
            $this->filesystem->delete($newPath);
        }
        // deplacement du fichier
        if (!$this->filesystem->has($newPath) && ($fh = fopen($oldPath, 'r+'))) {
            $result = $this->filesystem->writeStream($newPath, $fh);
            if (!fclose($fh) || !unlink($oldPath)) {
                trigger_error("old $oldPath cannot be remove", E_USER_WARNING);
            }
            return $result;
        }

        trigger_error("file $oldPath cannot be rename to $newPath", E_USER_WARNING);
        return true;
    }

    /**
     * @inheritdoc
     */
    public function getDestinationPath($filename) : string
    {
        $directory = '/' . substr($filename, 0, 2);
        return $directory;
    }

    /**
     * @inheritdoc
     * @throws \InvalidArgumentException
     * @throws \League\Flysystem\FileExistsException
     * @throws \League\Flysystem\FileNotFoundException
     * @throws \UnexpectedValueException
     */
    public function rename($filePath, $override = false) : string
    {
        // check du fichier dans le filesystem
        if (!is_file($filePath) || !is_readable($filePath)) {
            throw new \InvalidArgumentException("file $filePath does not exist or is not readable");
        }

        // fabrication du nouveau filename
        $newFilename = $this->createFilename($filePath);

        // creation du repertoire
        $directory = $this->createDirectory($newFilename);

        // deplacement du fichier
        $directory .= DIRECTORY_SEPARATOR . $newFilename;
        if ($this->execRename($filePath, $directory, $override)) {
            return $directory;
        }
        return $filePath;
    }
}
