<?php

namespace Move\Template\Html\Asset;

use Move\Analytics\Google\CommandBuilderInterface;
use Move\Utils\Str;

/**
 * Class GoogleAnalytics
 * @package Move\Template\Html\Asset
 */
class GoogleAnalytics implements AssetInterface
{

    /** @var array */
    private static $commands = [];

    /** @var string */
    private $gaCode;

    /** @var string */
    private $gaTestCode;

    /** @var null|string */
    private $scriptURL;

    /** @var bool */
    private $sendPageView;

    /**
     * GoogleAnalytics constructor.
     * @param string $gaCode
     * @param bool $sendPageView
     * @param string|null $scriptURL
     */
    public function __construct(string $gaCode, bool $sendPageView = true, string $scriptURL = null)
    {
        $this->gaCode = $gaCode;
        $this->scriptURL = $scriptURL ?: '//www.google-analytics.com/analytics.js';
        $this->sendPageView = $sendPageView;
    }

    /**
     * @param string $cmd
     * @param array|string|null $value
     * @param string|null $extra
     */
    public static function addCommand(string $cmd, $value = null, $extra = null) : void
    {
        $cmdArray = [$cmd, $value];
        if ($extra !== null) {
            $cmdArray[] = $extra;
        }
        self::$commands[] = $cmdArray;
    }

    /**
     * @param \Move\Analytics\Google\CommandBuilderInterface $builder
     */
    public static function applyCommandBuilder(CommandBuilderInterface $builder) : void
    {
        foreach ($builder->getCommands() as $args) {
            static::addCommand(...$args);
        }
    }

    /**
     * @param int $groupId
     * @param string $value
     */
    public static function setContentGroup($groupId, string $value)
    {
        self::addCommand('set', 'contentGroup' . $groupId, $value);
    }

    /**
     * Renvoi la copie sous forme de tableau de la metadata
     * @return array
     */
    public function getArrayCopy()
    {
        return [
            'cmds' => self::$commands,
            'code' => $this->gaCode,
            'scriptURL' => $this->scriptURL,
            'sendPageView' => $this->sendPageView,
        ];
    }

    /**
     * Effectue un rendu de l'objet en HTML
     * @return string
     */
    public function __toString()
    {
        $scriptUrl = $this->scriptURL;
        $code = $this->gaCode;
        $cmdScript = $script = '';
        foreach (self::$commands as $cmdOpt) {
            $cmd = array_shift($cmdOpt);
            $cmdScript .= "\tga('$cmd'";

            $action = array_shift($cmdOpt);
            if ($action !== null) {
                $cmdScript .= ',' . Str::toJson($action);
            }
            if (!empty($cmdOpt)) {
                $cmdScript .= ',"' . implode("','", $cmdOpt) . '"';
            }
            $cmdScript .= ");\n";
        }
        if ($this->sendPageView === true) {
            $cmdScript .= "\tga('send', 'pageview');\n";
        }
        $script .= <<<EOF
<script>
    (function(i,s,o,g,r,a,m){i['GoogleAnalyticsObject']=r;i[r]=i[r]||function(){
    (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),
    m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)
    })(window,document,'script','$scriptUrl','ga');
    ga('create', '$code', 'auto');
    $cmdScript
</script>
EOF;
        return $script;
    }

    /**
     * @param bool $sendPageView
     * @return GoogleAnalytics
     */
    public function setSendPageView(bool $sendPageView) : GoogleAnalytics
    {
        $this->sendPageView = $sendPageView;
        return $this;
    }

    /**
     * @param string $gtmId
     * @return $this
     */
    public function setGaTestCode(string $gtmId)
    {
        $this->gaTestCode = $gtmId;
        return $this;
    }

    /**
     * @return bool
     */
    public function isSendPageView() : bool
    {
        return $this->sendPageView;
    }

    /**
     * @param string $scriptURL
     * @return GoogleAnalytics
     */
    public function setScriptURL(string $scriptURL) : GoogleAnalytics
    {
        $this->scriptURL = $scriptURL;
        return $this;
    }

    /**
     * @return string
     */
    public function getScriptURL() : string
    {
        return $this->scriptURL;
    }

    /**
     * @param string $gaCode
     * @return GoogleAnalytics
     */
    public function setGaCode(string $gaCode) : GoogleAnalytics
    {
        $this->gaCode = $gaCode;
        return $this;
    }

    /**
     * @return string
     */
    public function getGaCode() : string
    {
        return $this->gaCode;
    }
}
