<?php

namespace Cms\Client;

use Cms\Model\AbstractAttributeModel;

/**
 * Class AttributeAwareModelTrait
 * @package Cms\Client
 * @property AbstractAttributeModel[] $attributes
 */
trait AttributeAwareModelTrait
{

    /** @var  array */
    protected $attributes;

    /**
     * @param AbstractAttributeModel|array $attributeData
     * @return AbstractAttributeModel
     */
    abstract public function handleAttribute($attributeData);

    /**
     * @param AbstractAttributeModel $attributeModel
     * @return $this
     */
    public function addAttribute(AbstractAttributeModel $attributeModel)
    {
        $this->attributes[] = $attributeModel;
        return $this;
    }

    /**
     * @param string $attrName
     * @throws \UnexpectedValueException
     */
    public function removeAttributes(string $attrName)
    {
        $this->attributes = array_values(
            array_filter(
                $this->getAttributes(),
                function ($attribute) use ($attrName) {
                    return $attribute->name !== $attrName;
                }
            )
        );
    }

    /**
     * @param string $attrName
     * @param string $value
     * @param null $quantity
     * @return AbstractAttributeModel
     * @throws \UnexpectedValueException
     */
    public function setAttribute($attrName, $value, $quantity = null)
    {
        $attrParams = [
            'name' => $attrName,
            'value' => $value,
        ];
        if (($attrModel = $this->getFirstAttributeByName($attrName))) {
            $attrModel->populate($attrParams);
        } else {
            $attrModel = $this->handleAttribute($attrParams);
            $this->addAttribute($attrModel);
        }
        if (isset($quantity)) {
            $attrModel->quantity = $quantity;
        }
        return $attrModel;
    }

    /**
     * @param string $attrName
     * @return AbstractAttributeModel[]
     * @throws \UnexpectedValueException
     */
    public function getAttributesByName($attrName)
    {
        return array_values(
            array_filter(
                $this->getAttributes(),
                function (AbstractAttributeModel $attr) use ($attrName) {
                    return $attr->name === $attrName;
                }
            )
        );
    }

    /**
     * @param string $attrName
     * @return AbstractAttributeModel|null
     * @throws \UnexpectedValueException
     */
    public function getFirstAttributeByName($attrName)
    {
        $attrs = $this->getAttributesByName($attrName);
        return !empty($attrs) ? $attrs[0] : null;
    }

    /**
     * @return AbstractAttributeModel[]
     * @throws \UnexpectedValueException
     */
    public function getAttributes()
    {
        $attributes = $this->attributes ?: [];
        foreach ($attributes as &$attribute) {
            if (!empty($attribute)) {
                $attribute = $this->convertAttribute($attribute);
            } else {
                $attribute = null;
            }
        }
        return array_values(array_filter($attributes));
    }

    /**
     * @param AbstractAttributeModel|array $attributeData
     * @return AbstractAttributeModel
     * @throws \UnexpectedValueException
     */
    private function convertAttribute($attributeData)
    {
        if ($attributeData instanceof AbstractAttributeModel) {
            return $attributeData;
        }
        $resource = $this->handleAttribute($attributeData);
        if (!$resource instanceof AbstractAttributeModel) {
            throw new \UnexpectedValueException('attribute incorrectly handled');
        }
        return $resource;
    }
}
