<?php

namespace Cms\Client\Member\Model;

use Cms\Model\Member\MemberAttributeModel;
use Cms\Model\Member\MemberModel;

/**
 * Class MemberRestfulModel
 * @package Cms\Client\Member\Model
 * @property MemberAttributeModel[] $attributes
 * @property string $password
 * @property int $password_scope_id
 * @property string $update_password
 * @property bool ask_reset_password
 * @property \Cms\Model\Member\MemberLoginModel[] $logins
 */
class MemberRestfulModel extends MemberModel
{

    /** @var MemberAttributeModel[] */
    protected $attributes;

    /** @var string */
    protected $password;

    /** @var int */
    protected $password_scope_id;

    /** @var string */
    protected $update_password;

    /** @var bool */
    protected $ask_reset_password;

    /** @var \Cms\Model\Member\MemberLoginModel[] */
    protected $logins;

    /**
     * @param null|int $scopeId
     * @return MemberAttributeModel[]
     */
    public function getAttributes($scopeId = null)
    {
        $attributes = $this->attributes ?: [];
        foreach ($attributes as &$attribute) {
            if (!empty($attribute)) {
                if (($scopeId === null) || ($scopeId !== null && $attribute->scope_id === $scopeId)) {
                    $attribute = $this->convertAttribute($attribute);
                } else {
                    $attribute = null;
                }
            } else {
                $attribute = null;
            }
        }
        return array_values(array_filter($attributes));
    }

    /**
     * @param MemberAttributeModel|array $attributeData
     * @return MemberAttributeModel
     */
    private function convertAttribute($attributeData)
    {
        if ($attributeData instanceof MemberAttributeModel) {
            return $attributeData;
        }
        $resource = $this->handleAttribute($attributeData);
        if (!$resource instanceof MemberAttributeModel) {
            throw new \UnexpectedValueException('attribute incorrectly handled');
        }
        return $resource;
    }

    /**
     * @param MemberAttributeModel|array $attributeData
     * @return MemberAttributeModel
     */
    public function handleAttribute($attributeData)
    {
        if (!$attributeData instanceof MemberAttributeModel) {
            $attributeData = (new MemberAttributeModel())->populate($attributeData, true);
        }
        return $attributeData;
    }

    /**
     * @param int $scopeId
     * @param string $name
     * @return MemberAttributeModel|null
     */
    public function getAttributeByName($scopeId, $name)
    {
        // récupération des anciens attributs
        $attributes = $this->getAttributes($scopeId);
        // check si l'attribut existe déjà
        if (!empty($attributes)) {
            foreach ($attributes as &$attribute) {
                if ($attribute->name == $name) {
                    return $attribute;
                }
            }
        }
        return null;
    }

    /**
     * @param int $scopeId
     * @param string $name
     * @return null|string
     */
    public function getAttributeValue($scopeId, $name)
    {
        $attr = $this->getAttributeByName($scopeId, $name);
        return $attr ? $attr->value : null;
    }

    /**
     * @param string $scopeId
     * @param string $name
     * @param string $value
     * @param bool $forceUpdate
     * @return MemberRestfulModel $this
     */
    public function setAttributes($scopeId, $name, $value, $forceUpdate = false)
    {
        // maj de l'existant si present
        $attribute = $this->getAttributeByName($scopeId, $name);
        if ($attribute !== null && $forceUpdate === true) {
            $attribute->value = $value;
        }

        // si l'attribut n'est pas présent
        if ($attribute === null) {
            // récupération des anciens attributs
            $attributes = $this->getAttributes();
            // ajout du nouvel attribut
            $attributes[] = $this->handleAttribute([
                'scope_id' => $scopeId,
                'name' => $name,
                'value' => $value,
            ]);
            // insertion dans l'objet
            $this->attributes = $attributes;
        }

        // renvoi
        return $this;
    }

    /**
     * @param int $scopeId
     * @return \Cms\Model\Member\MemberLoginModel|null
     */
    public function getLoginByScope($scopeId)
    {
        foreach ($this->logins ?: [] as $login) {
            if ((int)$login['scope_id'] === (int)$scopeId) {
                return $login;
            }
        }
        return null;
    }
}
