<?php


namespace Cms\Model;

use POM\DomainObjectInterface;
use Traversable;

/**
 * Class AbstractModelCollection
 * @package Cms\Model
 */
abstract class AbstractModelCollection implements \IteratorAggregate
{

    /** @var \POM\DomainObjectInterface[] */
    private $models = [];

    /**
     * @param array $modelData
     * @return \POM\DomainObjectInterface
     */
    abstract public function handleModel(array $modelData) : DomainObjectInterface;

    /**
     * @param DomainObjectInterface $model
     * @return $this
     */
    public function addModel(DomainObjectInterface $model)
    {
        $this->models[] = $model;
        return $this;
    }

    /**
     * @param array $dataset
     * @return $this
     */
    public function addItem(array $dataset)
    {
        $model = $this->handleModel($dataset);
        return $this->addModel($model);
    }

    /**
     * @param \POM\DomainObjectInterface|array $modelData
     * @return \POM\DomainObjectInterface
     * @throws \UnexpectedValueException
     */
    private function convertModel($modelData)
    {
        if ($modelData instanceof DomainObjectInterface) {
            return $modelData;
        }
        $model = $this->handleModel($modelData);
        if (!$model instanceof DomainObjectInterface) {
            throw new \UnexpectedValueException('model incorrectly handled');
        }
        return $model;
    }

    /**
     * @return \POM\DomainObjectInterface[]
     * @throws \UnexpectedValueException
     */
    public function getModels()
    {
        $models = $this->models ?: [];
        foreach ($models as &$model) {
            if (!empty($model)) {
                $model = $this->convertModel($model);
            } else {
                $model = null;
            }
        }
        return array_values(array_filter($models));
    }

    /**
     * Retrieve an external iterator
     * @link http://php.net/manual/en/iteratoraggregate.getiterator.php
     * @return Traversable An instance of an object implementing <b>Iterator</b> or
     * <b>Traversable</b>
     * @since 5.0.0
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->models);
    }
}
