<?php

namespace Move\ObjectMapper;

use NewcomDigital\Enum\Enum;
use POM\DomainObjectInterface;

/**
 * Class ObjectTransformer
 * @package Move\ObjectMapper
 */
class ObjectTransformer
{

    /** @var array */
    protected $modifiers = [];

    /** @var callable */
    protected $pipeTransformer;

    /** @var string */
    protected $dateFormat = \DateTime::W3C;


    /**
     * @param string $fieldName
     * @param callable $modifier
     * @param array $args
     * @return $this
     */
    public function modify($fieldName, callable $modifier, array $args = [])
    {
        $this->modifiers[$fieldName] = [$modifier, $args];

        return $this;
    }

    /**
     * @param callable $transformer
     * @return $this
     */
    public function pipe(callable $transformer)
    {
        $this->pipeTransformer = $transformer;

        return $this;
    }

    /**
     * @param mixed $object
     * @return array
     */
    public function __invoke($object)
    {
        return $this->transform($object);
    }

    /**
     * @param DomainObjectInterface $object
     * @return array
     */
    public function transform(DomainObjectInterface $object)
    {
        // convertion des champs
        $dataset = $object->getArrayCopy();
        $dataset = array_map([$this, 'mapObjectValue'], $dataset);

        // application des modifier
        $modifiers = array_intersect_key($this->modifiers, $dataset) ?: [];
        foreach ($modifiers as $fieldName => $modifier) {
            array_unshift($modifier[1], $dataset[$fieldName]);
            $dataset[$fieldName] = \call_user_func_array($modifier[0], $modifier[1]);
        }

        // execution du pipe
        if (\is_callable($this->pipeTransformer)) {
            $dataset = \call_user_func($this->pipeTransformer, $dataset);
        }

        return $dataset;
    }

    /**
     * @param string $dateFormat
     * @return ObjectTransformer
     */
    public function setDateFormat(string $dateFormat)
    {
        $this->dateFormat = $dateFormat;
        return $this;
    }

    /**
     * @param mixed $value
     * @return array|float|int|string
     */
    protected function mapObjectValue($value)
    {
        if ($value instanceof Enum) {
            return (string)$value;
        }
        if ($value instanceof \DateTime) {
            return $value->format($this->dateFormat);
        }
        if ($value instanceof DomainObjectInterface) {
            return (new self())
                ->setDateFormat($this->dateFormat)
                ->transform($value);
        }

        // recurse dans l'object
        if (\is_array($value)) {
            $value = array_map([$this, 'mapObjectValue'], $value);
        }

        if (is_numeric($value)) {
            if (\is_int($value)) {
                if (\strlen($value) !== \strlen((int)$value)) {
                    return $value;
                }
                return (int)$value;
            }
            if (\strlen($value) !== \strlen((float)$value)) {
                return $value;
            }
            return (float)$value;
        }
        return $value;
    }
}
