<?php

namespace Move\Search;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class AbstractSearchIndexHelper
 * @package Move\Search
 */
abstract class AbstractIndexHelper implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    /** @var \Move\Search\AbstractIndexFactory */
    protected $indexFactory;

    /**
     * AbstractIndexHelper constructor.
     * @param \Move\Search\AbstractIndexFactory $indexFactory
     */
    public function __construct(AbstractIndexFactory $indexFactory)
    {
        $this->indexFactory = $indexFactory;
    }

    /**
     * @return \Move\Search\ElasticMappingBuilderInterface
     */
    abstract protected function getMappingBuilder() : ElasticMappingBuilderInterface;

    /**
     * @param string|null $indexName
     * @return \Move\Search\ElasticDocumentHandler
     * @throws \UnexpectedValueException
     */
    public function getDocumentHandler(
        string $indexName = null
    ) : ElasticDocumentHandler {
        $index = $this->indexFactory->getIndex($indexName);
        // envoi du mapping
        $mappingBuilder = $this->getMappingBuilder();
        $mapping = $mappingBuilder->createMapping($index);
        $docHandler = new ElasticDocumentHandler($mapping);
        return $docHandler;
    }

    /**
     * @param \Move\Search\ElasticDocumentHandler $docHandler
     * @param \Elastica\Document[] $documents
     * @return int
     */
    protected function indexDocuments(ElasticDocumentHandler $docHandler, array $documents) : int
    {
        $count = 0;
        try {
            $docHandler->indexDocuments($documents);
            $count = count($documents);
        } catch (\Exception $e) {
            foreach ($documents as $document) {
                try {
                    $docHandler->indexDocument($document);
                    $count++;
                } catch (\Exception $e) {
                    $this->logger->error($e->getMessage(), [
                        'doc' => $document->toArray(),
                    ]);
                }
            }
        }
        $docHandler->refreshIndex();
        return $count;
    }
}
