<?php

namespace Move\Specification;

use Move\Utils\Reflection;
use PHPUnit\Framework\TestCase;

/**
 * Class PhpFilterVarSpecificationTest
 * @package Move\Specification
 */
class PhpFilterVarSpecificationTest extends TestCase
{
    /**
     * @return array
     */
    public static function provideIsSatisfiedBy()
    {
        /* [$filter, $options, $object, $expected] */
        return [
            'simple invalid url' => [FILTER_VALIDATE_URL, [], 'www.google.com', false],
            'incomplete url' => [FILTER_VALIDATE_URL, [], 'wwwgooglecom', false],
            'simple valid url' => [FILTER_VALIDATE_URL, [], 'http://www.google.com', true],

            'callable return true' => [
                FILTER_CALLBACK, ['options' => function($el) {return $el==='test'; }], 'test', true
            ],
            'callable return false' => [
                FILTER_CALLBACK, ['options' => function($el) {return $el!=='test'; }], 'test', false
            ],
            'valid array as value' => [
                FILTER_VALIDATE_URL, [], ['https://www.google.com', 'ftp://www.google.com'], true
            ],
            'invalid array as value' => [
                FILTER_VALIDATE_URL, [], ['https://www.google.com', 'wwwgooglecom'], false
            ],

        ];
    }

    /**
     * @covers PhpFilterVarSpecification::__construct
     * @return mixed
     */
    public function testConstructWithoutParams()
    {
        $className = PhpFilterVarSpecification::class;

        $instance = new $className();
        $this->assertInstanceOf($className, $instance);

        $optionsValue = Reflection::getValue($instance, 'filter');
        $this->assertEquals(FILTER_DEFAULT, $optionsValue);

        $optionsValue = Reflection::getValue($instance, 'options');
        $this->assertEquals([], $optionsValue);

        return $instance;
    }

    /**
     * @covers PhpFilterVarSpecification::__construct
     * @return PhpFilterVarSpecification
     */
    public function testConstruct()
    {
        $className = PhpFilterVarSpecification::class;

        $instance = new $className(FILTER_VALIDATE_URL);
        $this->assertInstanceOf($className, $instance);

        $optionsValue = Reflection::getValue($instance, 'filter');
        $this->assertEquals(FILTER_VALIDATE_URL, $optionsValue);

        $optionsValue = Reflection::getValue($instance, 'options');
        $this->assertEquals([], $optionsValue);

        return $instance;
    }

    /**
     * @covers PhpFilterVarSpecification::isSatisfiedBy
     * @dataProvider ProvideIsSatisfiedBy
     * @depends testConstruct
     * @param $filter
     * @param $options
     * @param $object
     * @param $expected
     */
    public function testIsSatisfiedBy($filter, $options, $object, $expected)
    {
        $spec = new PhpFilterVarSpecification($filter, $options);
        $this->assertEquals($expected, $spec->isSatisfiedBy($object));
    }
}
