<?php

namespace Cms\Restful;

use League\Container\Argument\RawArgument;
use League\Container\ServiceProvider\AbstractServiceProvider;
use Move\Command\Command\CreateCommand;
use Move\Command\Command\DeleteCommand;
use Move\Command\Command\FetchAllCommand;
use Move\Command\Command\FetchCommand;
use Move\Command\Command\UpdateCommand;
use Move\Command\CommandBusInterface;
use Move\Command\TacticianBusFactory;
use Move\Filter\Validator\StateValidatorInterface;
use Move\ObjectMapper\ObjectTransformer;
use POM\Service\AdapterInterface;

/**
 * Class AbstractResourceProvider
 * @package Cms\Restful
 */
abstract class AbstractResourceProvider extends AbstractServiceProvider
{

    /** @var array */
    protected $default_cmds = [
        CreateCommand::class,
        UpdateCommand::class,
        DeleteCommand::class,
        FetchCommand::class,
        FetchAllCommand::class,
    ];

    /**
     * @param string $cmdHandlerClass
     * @param array $commandList
     * @param array $middleware
     * @return \Move\Command\CommandBusInterface
     * @throws \LogicException
     */
    protected function createCommandBus(
        $cmdHandlerClass,
        array $commandList = [],
        array $middleware = []
    ) {
        if (!$this->container->has($cmdHandlerClass)) {
            throw new \LogicException("the command handler $cmdHandlerClass must register "
                . 'in container before create command bus');
        }
        // cmd bus
        $bus = TacticianBusFactory::mapToClassHandler(
            $this->container,
            $commandList ?: $this->default_cmds,
            $cmdHandlerClass,
            $middleware
        );
        return $bus;
    }

    /**
     * @param string $cmdHandlerClass
     * @param string $mapperClass
     * @param StateValidatorInterface $validator
     */
    protected function registerCommandHandler(
        $cmdHandlerClass,
        $mapperClass,
        $validator = null
    ) {
        // cmd handler
        $this->container->add($cmdHandlerClass)
            ->withArguments([
                $mapperClass,
            ]);

        // ajoute le validateur
        if (null !== $validator) {
            $this->container->inflector($cmdHandlerClass)
                ->invokeMethod('setValidator', [
                    new RawArgument($validator),
                ]);
        }
    }

    /**
     * @param string $serviceClass
     * @param string $mapperClass
     * @param string $cmdHandlerClass
     * @param null|string $transformerClass
     * @param array|\Move\Command\CommandBusInterface $commandList
     * @param array $middleware
     * @throws \LogicException
     */
    protected function registerService(
        $serviceClass,
        $mapperClass,
        $cmdHandlerClass,
        $transformerClass = null,
        $commandList = [],
        array $middleware = []
    ) {
        // cmd bus
        if ($commandList instanceof CommandBusInterface) {
            $commandBus = $commandList;
        } else {
            $commandBus = $this->createCommandBus($cmdHandlerClass, $commandList, $middleware);
        }

        // transformer
        if (!empty($transformerClass)) {
            $this->container->add($transformerClass);
        }
        // add service
        $this->container->add($serviceClass)
            ->withArguments([
                $mapperClass,
                new RawArgument($commandBus),
                $transformerClass ?: new RawArgument(new ObjectTransformer()),
            ]);
    }

    /**
     * @param string|array $mapperClass
     * @param string|null $adapterClass
     * @return void
     */
    protected function registerAdapter($mapperClass, string $adapterClass = null)
    {
        if (!empty($mapperClass) && !\is_array($mapperClass)) {
            $mapperClass = [$mapperClass];
        }
        // mapper
        foreach ($mapperClass as $class) {
            $this->container->add($class)
                ->withArgument($adapterClass ?: AdapterInterface::class);
        }
    }

    /**
     * @param string $serviceClass
     * @param string $mapperClass
     * @param string $cmdHandlerClass
     * @param StateValidatorInterface $validator
     * @param null|string $transformerClass
     * @param array|\Move\Command\CommandBusInterface $commandList
     * @param array $middleware
     * @throws \LogicException
     */
    protected function registerResource(
        $serviceClass,
        $mapperClass,
        $cmdHandlerClass,
        StateValidatorInterface $validator = null,
        $transformerClass = null,
        $commandList = [],
        array $middleware = []
    ) {
        // db adapter
        $this->registerAdapter($mapperClass);

        // cmd handler
        $this->registerCommandHandler($cmdHandlerClass, $mapperClass, $validator);

        // service
        $this->registerService(
            $serviceClass,
            $mapperClass,
            $cmdHandlerClass,
            $transformerClass,
            $commandList,
            $middleware
        );
    }


    /**
     * @param string $serviceClass
     * @param string $mapperClass
     * @param string $cmdHandlerClass
     * @param null|string $transformerClass
     * @param array $commandList
     * @param array $middleware
     * @deprecated
     * @throws \LogicException
     */
    protected function serviceFactory(
        $serviceClass,
        $mapperClass,
        $cmdHandlerClass,
        $transformerClass = null,
        array $commandList = [],
        array $middleware = []
    ) {
        $this->registerService(
            $serviceClass,
            $mapperClass,
            $cmdHandlerClass,
            $transformerClass,
            $commandList,
            $middleware
        );
    }
}
