<?php


namespace Move\Http;

use Dflydev\FigCookies\FigRequestCookies;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Message\UploadedFileInterface;

/**
 * Class RequestUtils
 * @package Move\Http
 */
class RequestUtils
{
    /**
     * @param ServerRequestInterface $request
     * @param string $paramName
     * @param null|mixed $defaultValue
     * @return mixed
     */
    public static function getQueryValue(ServerRequestInterface $request, $paramName, $defaultValue = null)
    {
        $queryParams = $request->getQueryParams();
        if (isset($queryParams[$paramName])) {
            return $queryParams[$paramName];
        }
        return $defaultValue;
    }

    /**
     * @param ServerRequestInterface $request
     * @param string $mime
     * @return bool
     */
    public static function isAcceptable(ServerRequestInterface $request, $mime) : bool
    {
        // filtrage sur le header accept
        $acceptType = $request->getHeaderLine('Accept');
        // check du type
        return preg_match('@' . preg_quote($mime, '@') . '(\+[a-z]+)?(,|;|$)@', $acceptType) === 1;
    }

    /**
     * @param ServerRequestInterface $request
     * @param string $name
     * @return null|string
     */
    public static function GetCookieValue(ServerRequestInterface $request, $name)
    {
        $cookies = $request->getCookieParams();
        if (!isset($cookies[$name])) {
            return FigRequestCookies::get($request, $name)->getValue();
        }
        return $cookies[$name];
    }

    /**
     * @param array $dataset
     * @return array
     */
    public static function ConvertArrayToMultipartArray(array $dataset) : array
    {
        // convertion du parsing en multipart
        $multiPartBody = [];
        foreach ($dataset as $fieldName => $value) {
            if (\is_array($value)) {
                foreach ($value as $subName => $content) {
                    $multiPartBody[] = [
                        'contents' => $content,
                        'name' => $fieldName . (!is_numeric($subName) ? '[' . $subName . ']' : '[]'),
                    ];
                }
            } else {
                $multiPartBody[] = [
                    'contents' => $value,
                    'name' => $fieldName,
                ];
            }
        }
        return $multiPartBody;
    }

    /**
     * @param ServerRequestInterface $request
     * @return array
     * @throws \RuntimeException
     */
    public static function ConvertBodyToMultipartArray(ServerRequestInterface $request) : array
    {
        // récup des infos de la request
        $bodyParsed = $request->getParsedBody() ?: [];
        $uploadedFiles = $request->getUploadedFiles() ?: [];

        // convertion du parsing en multipart
        $multiPartBody = self::ConvertArrayToMultipartArray($bodyParsed);

        // ajout des upload files
        foreach ($uploadedFiles as $fieldName => $file) {
            /** @var UploadedFileInterface $file */
            $multiPartBody[] = [
                'filename' => $file->getClientFilename(),
                'name' => $fieldName,
                'contents' => $file->getStream(),
            ];
        }
        return $multiPartBody;
    }

    /**
     * @param ServerRequestInterface $request
     * @return string
     */
    public static function getIpClient(ServerRequestInterface $request) : string
    {
        $serverParams = $request->getServerParams();
        if (isset($serverParams['HTTP_X_FORWARDED_FOR'])) {
            $forwardedIP = explode(',', $serverParams['HTTP_X_FORWARDED_FOR']);
            return trim($forwardedIP[0]);
        }
        return $serverParams['REMOTE_ADDR'];
    }
}
